<?php

namespace App\Http\Controllers;

use App\Models\Education;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class EducationController extends Controller
{
    // List all educations for a student
    public function index($student_id)
    {
        $this->authorize('viewAny', [Education::class, $student_id]);
        $educations = Education::where('student_id', $student_id)->get();
        return response()->json(['educations' => $educations]);
    }

    // Show a single education
    public function show($id)
    {
        $education = Education::findOrFail($id);
        $this->authorize('view', $education);
        return response()->json($education);
    }

    // Create a new education
    public function store(Request $request, $student_id)
    {
        $this->authorize('create', [Education::class, $student_id]);
        $validated = $request->validate([
            'school' => 'required|string|max:255',
            'degree' => 'nullable|string|max:255',
            'field' => 'nullable|string|max:255',
            'start_year' => 'nullable|string|max:10',
            'end_year' => 'nullable|string|max:10',
            'description' => 'nullable|string',
        ]);
        $validated['id'] = (string) Str::uuid();
        $validated['student_id'] = $student_id;
        $education = Education::create($validated);
        return response()->json($education, 201);
    }

    // Update an education
    public function update(Request $request, $id)
    {
        $education = Education::findOrFail($id);
        $this->authorize('update', $education);
        $validated = $request->validate([
            'school' => 'sometimes|required|string|max:255',
            'degree' => 'nullable|string|max:255',
            'field' => 'nullable|string|max:255',
            'start_year' => 'nullable|string|max:10',
            'end_year' => 'nullable|string|max:10',
            'description' => 'nullable|string',
        ]);
        $education->update($validated);
        return response()->json($education);
    }

    // Delete an education
    public function destroy($id)
    {
        $education = Education::findOrFail($id);
        $this->authorize('delete', $education);
        $education->delete();
        return response()->json(['message' => 'Education deleted']);
    }

    // Batch create educations
    public function storeBatch(Request $request, $student_id)
    {
        $this->authorize('create', [Education::class, $student_id]);
        
        $request->validate([
            'educations' => 'required|array|max:10',
            'educations.*.school' => 'required|string|max:255',
            'educations.*.degree' => 'nullable|string|max:255',
            'educations.*.field' => 'nullable|string|max:255',
            'educations.*.start_year' => 'nullable|string|max:10',
            'educations.*.end_year' => 'nullable|string|max:10',
            'educations.*.description' => 'nullable|string',
        ]);

        $educations = collect($request->educations)->map(function ($education) use ($student_id) {
            return [
                'id' => (string) Str::uuid(),
                'student_id' => $student_id,
                'school' => $education['school'],
                'degree' => $education['degree'] ?? null,
                'field' => $education['field'] ?? null,
                'start_year' => $education['start_year'] ?? null,
                'end_year' => $education['end_year'] ?? null,
                'description' => $education['description'] ?? null,
                'created_at' => now(),
                'updated_at' => now(),
            ];
        });

        Education::insert($educations->toArray());

        return response()->json([
            'message' => 'Educations created successfully',
            'count' => $educations->count()
        ], 201);
    }
} 