<?php

namespace App\Http\Controllers;

use App\Models\Experience;
use App\Models\Student;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ExperienceController extends Controller
{
    // List all experiences for a student
    public function index($student_id)
    {
        $this->authorize('viewAny', [Experience::class, $student_id]);
        $experiences = Experience::where('student_id', $student_id)->get();
        return response()->json(['experiences' => $experiences]);
    }

    // Show a single experience
    public function show($id)
    {
        $experience = Experience::findOrFail($id);
        $this->authorize('view', $experience);
        return response()->json($experience);
    }

    // Create a new experience
    public function store(Request $request, $student_id)
    {
        $this->authorize('create', [Experience::class, $student_id]);
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'company' => 'nullable|string|max:255',
            'start_date' => 'nullable|string', // Changed to string to handle partial dates
            'end_date' => 'nullable|string',   // Changed to string to handle partial dates
            'description' => 'nullable|string',
        ]);
        
        // Normalize dates
        $validated['start_date'] = $this->normalizeDate($validated['start_date'] ?? null);
        $validated['end_date'] = $this->normalizeDate($validated['end_date'] ?? null);
        
        $validated['id'] = (string) Str::uuid();
        $validated['student_id'] = $student_id;
        $experience = Experience::create($validated);
        return response()->json($experience, 201);
    }

    // Create multiple experiences at once
    public function storeBatch(Request $request, $student_id)
    {
        $this->authorize('create', [Experience::class, $student_id]);
        
        $validated = $request->validate([
            'experiences' => 'required|array|max:20', // Limit to 20 experiences for safety
            'experiences.*.title' => 'required|string|max:255',
            'experiences.*.company' => 'nullable|string|max:255',
            'experiences.*.start_date' => 'nullable|string', // Changed to string to handle partial dates
            'experiences.*.end_date' => 'nullable|string',   // Changed to string to handle partial dates
            'experiences.*.description' => 'nullable|string',
        ]);

        $experiencesData = [];
        foreach ($validated['experiences'] as $experienceData) {
            // Process and normalize dates
            $startDate = $this->normalizeDate($experienceData['start_date'] ?? null);
            $endDate = $this->normalizeDate($experienceData['end_date'] ?? null);
            
            $experiencesData[] = [
                'id' => (string) Str::uuid(),
                'student_id' => $student_id,
                'title' => $experienceData['title'],
                'company' => $experienceData['company'] ?? null,
                'start_date' => $startDate,
                'end_date' => $endDate,
                'description' => $experienceData['description'] ?? null,
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        // Use batch insert for better performance
        Experience::insert($experiencesData);
        
        // Return the created experiences
        $createdExperiences = Experience::where('student_id', $student_id)
            ->whereIn('title', array_column($validated['experiences'], 'title'))
            ->get();

        return response()->json([
            'message' => 'Experiences created successfully',
            'count' => count($experiencesData),
            'experiences' => $createdExperiences
        ], 201);
    }

    // Update an experience
    public function update(Request $request, $id)
    {
        $experience = Experience::findOrFail($id);
        $this->authorize('update', $experience);
        $validated = $request->validate([
            'title' => 'sometimes|required|string|max:255',
            'company' => 'nullable|string|max:255',
            'start_date' => 'nullable|string', // Changed to string to handle partial dates
            'end_date' => 'nullable|string',   // Changed to string to handle partial dates
            'description' => 'nullable|string',
        ]);
        
        // Normalize dates if provided
        if (isset($validated['start_date'])) {
            $validated['start_date'] = $this->normalizeDate($validated['start_date']);
        }
        if (isset($validated['end_date'])) {
            $validated['end_date'] = $this->normalizeDate($validated['end_date']);
        }
        
        $experience->update($validated);
        return response()->json($experience);
    }

    // Delete an experience
    public function destroy($id)
    {
        $experience = Experience::findOrFail($id);
        $this->authorize('delete', $experience);
        $experience->delete();
        return response()->json(['message' => 'Experience deleted']);
    }
    
    /**
     * Normalize date input to valid date format
     * Handles partial dates like "2024-07" or "2024"
     */
    private function normalizeDate($dateInput)
    {
        if (empty($dateInput)) {
            return null;
        }
        
        // If it's already a valid full date, return it
        if (\DateTime::createFromFormat('Y-m-d', $dateInput)) {
            return $dateInput;
        }
        
        // Handle year-month format (e.g., "2024-07")
        if (preg_match('/^\d{4}-\d{2}$/', $dateInput)) {
            return $dateInput . '-01'; // Add first day of month
        }
        
        // Handle year only format (e.g., "2024")
        if (preg_match('/^\d{4}$/', $dateInput)) {
            return $dateInput . '-01-01'; // Add January 1st
        }
        
        // Try to parse other date formats
        try {
            $date = new \DateTime($dateInput);
            return $date->format('Y-m-d');
        } catch (\Exception $e) {
            // If all else fails, return null
            return null;
        }
    }
} 