<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Shortlist;
use App\Models\Student;
use App\Models\SkillNormalization;
use App\Services\AISkillNormalizer;
use Illuminate\Support\Str;

class ShortlistController extends Controller
{
    // List all students in the recruiter's shortlist
    public function index(Request $request)
    {
        $recruiterId = Auth::id();
        \Log::info('Shortlist request - Recruiter ID: ' . $recruiterId);
        
        $shortlistItems = Shortlist::with(['student.skills', 'student.educations', 'student.user'])
            ->where('recruiter_id', $recruiterId)
            ->get();
            
        \Log::info('Shortlist items count: ' . $shortlistItems->count());
        
        $shortlist = $shortlistItems->map(function ($item) {
                $photoUrl = $item->student->photo_url ?? null;
                // Convert relative photo URL to full URL if it exists
                if ($photoUrl && !str_starts_with($photoUrl, 'http')) {
                    $photoUrl = url($photoUrl);
                }
                
                $skills = [];
                $normalizedSkills = [];
                if ($item->student && $item->student->skills) {
                    $skills = $item->student->skills->map(function($skill) {
                        return [
                            'id' => $skill->id,
                            'name' => $skill->name,
                            'level' => $skill->level ?? null
                        ];
                    })->toArray();

                    $normalizedSkills = $this->normalizeSkills($item->student->skills);
                }
                
                $educations = [];
                if ($item->student && $item->student->educations) {
                    $educations = $item->student->educations->map(function($education) {
                        return [
                            'id' => $education->id,
                            'school' => $education->school,
                            'degree' => $education->degree ?? null,
                            'field' => $education->field ?? null,
                            'start_year' => $education->start_year ?? null,
                            'end_year' => $education->end_year ?? null,
                            'description' => $education->description ?? null
                        ];
                    })->toArray();
                }
                
                return [
                    'student_id' => $item->student_id,
                    'full_name' => $item->student->full_name ?? '',
                    'university' => $item->student->university ?? '',
                    'degree' => $item->student->degree ?? '',
                    'skills' => $skills,
                    'normalized_skills' => array_values($normalizedSkills),
                    'normalized_skill_names' => array_values(array_unique(array_map(fn($s) => $s['name'], $normalizedSkills))),
                    'educations' => $educations,
                    'private_note' => $item->private_note ?? '',
                    'photo_url' => $photoUrl,
                    'email' => $item->student->user->email ?? null,
                    'created_at' => $item->created_at,
                ];
            });
            
        \Log::info('Processed shortlist data: ' . json_encode($shortlist));
        return response()->json(['shortlist' => $shortlist]);
    }

    // Update the private note for a student
    public function updateNote(Request $request, $student_id)
    {
        $recruiterId = Auth::id();
        $shortlist = Shortlist::where('recruiter_id', $recruiterId)
            ->where('student_id', $student_id)
            ->firstOrFail();
        $request->validate([
            'private_note' => 'nullable|string',
        ]);
        $shortlist->private_note = $request->private_note;
        $shortlist->save();
        return response()->json(['message' => 'Note updated', 'shortlist' => $shortlist]);
    }

    // Add a student to the recruiter's shortlist
    public function store(Request $request)
    {
        $recruiterId = Auth::id();
        $request->validate([
            'student_id' => 'required|exists:students,user_id',
            'private_note' => 'nullable|string',
        ]);
        // Prevent duplicate shortlist entries
        $existing = Shortlist::where('recruiter_id', $recruiterId)
            ->where('student_id', $request->student_id)
            ->first();
        if ($existing) {
            return response()->json(['message' => 'Student already in shortlist'], 409);
        }
        $shortlist = Shortlist::create([
            'recruiter_id' => $recruiterId,
            'student_id' => $request->student_id,
            'private_note' => $request->private_note,
        ]);
        // Log activity for the student
        \App\Models\ActivityLog::create([
            'user_id' => $request->student_id,
            'action' => 'Shortlisted by recruiter',
        ]);
        return response()->json(['message' => 'Student added to shortlist', 'shortlist' => $shortlist], 201);
    }

    // Remove a student from the recruiter's shortlist
    public function destroy($student_id)
    {
        $recruiterId = Auth::id();
        $shortlist = Shortlist::where('recruiter_id', $recruiterId)
            ->where('student_id', $student_id)
            ->first();
        if (!$shortlist) {
            return response()->json(['message' => 'Not found'], 404);
        }
        $shortlist->delete();
        return response()->json(['message' => 'Student removed from shortlist']);
    }

    // Export shortlist to CSV
    public function exportCsv()
    {
        $recruiterId = Auth::id();
        $shortlistItems = Shortlist::with(['student.skills', 'student.educations', 'student.user'])
            ->where('recruiter_id', $recruiterId)
            ->get();

        // Prepare CSV data
        $headers = [
            'Full Name',
            'Email',
            'Phone',
            'University',
            'Degree',
            'Skills',
            'Private Note',
            'Added Date'
        ];

        // Create CSV content
        $csv = implode(',', array_map('addslashes', $headers)) . "\n";

        foreach ($shortlistItems as $item) {
            $normalizedSkills = $this->normalizeSkills($item->student->skills);
            $skills = !empty($normalizedSkills)
                ? collect($normalizedSkills)->pluck('name')->unique()->implode('; ')
                : $item->student->skills->pluck('name')->implode('; ');
            $createdDate = is_string($item->created_at) ? $item->created_at : $item->created_at->format('Y-m-d');
            $row = [
                $item->student->full_name ?? '',
                $item->student->user->email ?? '',
                $item->student->phone ?? '',
                $item->student->university ?? '',
                $item->student->degree ?? '',
                $skills,
                $item->private_note ?? '',
                $createdDate
            ];
            $csv .= implode(',', array_map('addslashes', $row)) . "\n";
        }

        $filename = 'shortlist_' . date('Y-m-d_His') . '.csv';

        return response(
            $csv,
            200,
            [
                'Content-Type' => 'text/csv; charset=utf-8',
                'Content-Disposition' => "attachment; filename=\"{$filename}\"",
            ]
        );
    }

    // Export shortlist to PDF
    public function exportPdf()
    {
        $recruiterId = Auth::id();
        $shortlistItems = Shortlist::with(['student.skills', 'student.educations', 'student.user'])
            ->where('recruiter_id', $recruiterId)
            ->get();

        // Prepare data for PDF
        $shortlist = $shortlistItems->map(function ($item) {
            $normalizedSkills = $this->normalizeSkills($item->student->skills);
            $skills = !empty($normalizedSkills)
                ? collect($normalizedSkills)->pluck('name')->unique()->implode(', ')
                : $item->student->skills->pluck('name')->implode(', ');
            $createdDate = is_string($item->created_at) ? $item->created_at : $item->created_at->format('Y-m-d');
            return [
                'full_name' => $item->student->full_name ?? 'N/A',
                'email' => $item->student->user->email ?? 'N/A',
                'phone' => $item->student->phone ?? 'N/A',
                'university' => $item->student->university ?? 'N/A',
                'degree' => $item->student->degree ?? 'N/A',
                'skills' => $skills,
                'private_note' => $item->private_note ?? '',
                'created_at' => $createdDate
            ];
        });

        // Create HTML for PDF
        $html = '
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <title>Shortlist Export</title>
            <style>
                body { font-family: Arial, sans-serif; margin: 20px; color: #333; }
                h1 { color: #2563eb; border-bottom: 3px solid #2563eb; padding-bottom: 10px; }
                table { width: 100%; border-collapse: collapse; margin-top: 20px; }
                th { background-color: #1f2937; color: white; padding: 12px; text-align: left; font-weight: bold; }
                td { padding: 10px; border-bottom: 1px solid #ddd; }
                tr:hover { background-color: #f3f4f6; }
                .skills { color: #7c3aed; font-weight: 600; }
                .note { font-style: italic; color: #666; }
                .footer { margin-top: 30px; font-size: 12px; color: #999; border-top: 1px solid #ddd; padding-top: 10px; }
            </style>
        </head>
        <body>
            <h1>📋 Talent Shortlist Export</h1>
            <p><strong>Exported on:</strong> ' . date('F d, Y H:i') . '</p>
            <p><strong>Total Candidates:</strong> ' . count($shortlist) . '</p>
            
            <table>
                <thead>
                    <tr>
                        <th>Name</th>
                        <th>Email</th>
                        <th>Phone</th>
                        <th>University</th>
                        <th>Degree</th>
                        <th>Skills</th>
                        <th>Notes</th>
                    </tr>
                </thead>
                <tbody>
        ';

        foreach ($shortlist as $candidate) {
            $html .= '
                    <tr>
                        <td><strong>' . htmlspecialchars($candidate['full_name']) . '</strong></td>
                        <td>' . htmlspecialchars($candidate['email']) . '</td>
                        <td>' . htmlspecialchars($candidate['phone']) . '</td>
                        <td>' . htmlspecialchars($candidate['university']) . '</td>
                        <td>' . htmlspecialchars($candidate['degree']) . '</td>
                        <td><span class="skills">' . htmlspecialchars($candidate['skills']) . '</span></td>
                        <td><span class="note">' . htmlspecialchars($candidate['private_note']) . '</span></td>
                    </tr>
            ';
        }

        $html .= '
                </tbody>
            </table>
            
            <div class="footer">
                <p>This document was automatically generated by PopCareers Recruitment System.</p>
            </div>
        </body>
        </html>
        ';

        // For now, return HTML. In production, you'd use a library like dompdf
        $filename = 'shortlist_' . date('Y-m-d_His') . '.html';

        return response(
            $html,
            200,
            [
                'Content-Type' => 'text/html; charset=utf-8',
                'Content-Disposition' => "attachment; filename=\"{$filename}\"",
            ]
        );
    }

    /**
     * Normalize skills into canonical buckets (e.g., JS -> JavaScript).
     * Uses database for persistence, falls back to in-memory map.
     */
    private function normalizeSkills($skills)
    {
        // Build map from database or fallback to default
        $map = $this->getSkillNormalizationMap();

        $normalized = [];

        foreach ($skills as $skill) {
            $raw = trim($skill->name ?? '');
            if ($raw === '') {
                continue;
            }

            $canonical = $this->canonicalSkillName($raw, $map);

            if (!isset($normalized[$canonical])) {
                $normalized[$canonical] = [
                    'name' => $canonical,
                    'aliases' => [],
                ];
            }

            // Track the original alias
            $normalized[$canonical]['aliases'][] = $raw;
        }

        // Ensure unique aliases
        foreach ($normalized as &$entry) {
            $entry['aliases'] = array_values(array_unique($entry['aliases']));
        }

        return $normalized;
    }

    private function getSkillNormalizationMap()
    {
        $map = [];
        
        // Load from database if available
        $normalizations = \App\Models\SkillNormalization::all();
        if ($normalizations->isNotEmpty()) {
            foreach ($normalizations as $norm) {
                $map[$norm->canonical_name] = $norm->aliases ?? [];
            }
            return $map;
        }

        // Fallback: In-memory map (default normalization rules)
        return $this->getDefaultSkillMap();
    }

    private function getDefaultSkillMap()
    {
        return [
            // Frontend
            'JavaScript' => ['js', 'javascript', 'java script', 'es6', 'es2015', 'es7', 'es8', 'es2016', 'ecmascript'],
            'TypeScript' => ['typescript', 'ts', 'tsx'],
            'React' => ['react', 'reactjs', 'react.js'],
            'Vue.js' => ['vue', 'vuejs', 'vue.js', 'vue 3'],
            'Angular' => ['angular', 'angularjs', 'angular.js', 'angular 14', 'angular 15'],
            'Svelte' => ['svelte', 'sveltekit'],
            'HTML/CSS' => ['html', 'css', 'html/css', 'html5', 'css3', 'front end', 'frontend', 'html + css'],
            'Tailwind' => ['tailwind', 'tailwind css', 'tailwindcss'],
            'Bootstrap' => ['bootstrap', 'bootstrap 5'],
            'Webpack' => ['webpack', 'webpack 5'],
            'Vite' => ['vite', 'vitejs'],

            // Backend
            'Node.js' => ['node', 'nodejs', 'node.js'],
            'Express.js' => ['express', 'expressjs', 'express.js'],
            'Python' => ['python', 'py'],
            'Django' => ['django', 'djangorestframework'],
            'Flask' => ['flask', 'flask restful'],
            'Java' => ['java', 'j2ee'],
            'Spring Boot' => ['spring', 'spring boot', 'springboot'],
            'C#' => ['c#', 'csharp', 'c sharp'],
            '.NET' => ['.net', 'dotnet', 'asp.net', 'asp.net core', '.net core'],
            'PHP' => ['php', 'php 7', 'php 8'],
            'Laravel' => ['laravel', 'laravel 9', 'laravel 10'],
            'Go' => ['go', 'golang', 'go lang'],
            'Rust' => ['rust', 'rustlang'],

            // Databases
            'SQL' => ['sql', 'mysql', 'postgres', 'postgresql', 'postgres sql', 'oracle sql', 'mariadb', 'sql server'],
            'MongoDB' => ['mongodb', 'mongo', 'nosql'],
            'Redis' => ['redis', 'cache'],
            'Firebase' => ['firebase', 'firestore', 'realtime database'],

            // AI/ML/Data
            'Machine Learning' => ['ml', 'machine learning', 'supervised learning', 'unsupervised learning'],
            'Deep Learning' => ['dl', 'deep learning', 'neural networks', 'cnn', 'rnn', 'lstm'],
            'TensorFlow' => ['tensorflow', 'tf'],
            'PyTorch' => ['pytorch', 'torch'],
            'Scikit-learn' => ['scikit-learn', 'sklearn'],
            'Data Analysis' => ['data analysis', 'statistical analysis'],
            'Data Visualization' => ['data visualization', 'tableau', 'power bi', 'matplotlib', 'seaborn'],

            // DevOps/Cloud
            'AWS' => ['aws', 'amazon web services', 'amazon aws'],
            'Google Cloud' => ['gcp', 'google cloud', 'google cloud platform'],
            'Azure' => ['azure', 'microsoft azure'],
            'Docker' => ['docker', 'containers', 'containerization'],
            'Kubernetes' => ['kubernetes', 'k8s', 'k8'],
            'CI/CD' => ['ci/cd', 'continuous integration', 'continuous deployment', 'jenkins', 'github actions', 'gitlab ci'],

            // Tools & Skills
            'Git' => ['git', 'version control', 'github', 'gitlab', 'bitbucket'],
            'Linux' => ['linux', 'ubuntu', 'centos'],
            'REST API' => ['rest', 'rest api', 'restful', 'api design'],
            'GraphQL' => ['graphql', 'gql'],
            'Web Development' => ['web dev', 'web development', 'web developer', 'fullstack', 'full stack', 'full-stack'],
            'Problem Solving' => ['problem solving', 'algorithms', 'dsa', 'data structures'],
            'Communication' => ['communication', 'soft skills', 'team work'],
            'Project Management' => ['project management', 'agile', 'scrum', 'kanban'],
        ];
    }

    private function canonicalSkillName(string $raw, array $map): string
    {
        // Create slug: lowercase, replace special chars, trim spaces
        $slug = strtolower(preg_replace('/\s+/', ' ', str_replace(['.', '_', '-'], ' ', trim($raw))));

        // Try database/hardcoded match first
        foreach ($map as $canonical => $aliases) {
            foreach ($aliases as $alias) {
                $aliasSlug = strtolower(preg_replace('/\s+/', ' ', str_replace(['.', '_', '-'], ' ', trim($alias))));
                if ($slug == $aliasSlug) {
                    return $canonical;
                }
            }
        }

        // Not found in database - try Groq AI fallback
        $groqResult = AISkillNormalizer::normalizeWithGroq($raw);
        if ($groqResult && isset($groqResult['canonical_name'])) {
            return $groqResult['canonical_name'];
        }

        // Default: Title Case of cleaned raw
        return Str::of($raw)->trim()->title();
    }
}