<?php

namespace App\Http\Controllers;

use App\Models\Student;
use App\Models\ActivityLog;
use App\Services\SmartSearchService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class SmartSearchController extends Controller
{
    private $smartSearchService;

    public function __construct(SmartSearchService $smartSearchService)
    {
        $this->smartSearchService = $smartSearchService;
    }

    /**
     * Smart search for students using natural language query
     * POST /api/students/smart-search
     */
    public function search(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'query' => 'required|string|min:3|max:500',
            'page' => 'integer|min:1',
            'per_page' => 'integer|min:1|max:50',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $query = $request->input('query');
            $page = $request->input('page', 1);
            $perPage = $request->input('per_page', 20);

            // Step 1: Extract structured criteria from natural language
            $criteria = $this->smartSearchService->extractCriteria($query);

            // Step 2: Pre-filter students from database based on extracted criteria
            $studentsQuery = Student::with([
                'user',
                'skills',
                'educations' => function($q) {
                    $q->orderBy('start_year', 'desc');
                },
                'experiences' => function($q) {
                    $q->orderBy('start_date', 'desc');
                }
            ]);

            // Apply filters if criteria were extracted
            if (!empty($criteria['skills'])) {
                $studentsQuery->whereHas('skills', function($q) use ($criteria) {
                    $q->whereIn('name', $criteria['skills']);
                });
            }

            if ($criteria['university']) {
                $studentsQuery->whereHas('educations', function($q) use ($criteria) {
                    $q->where('school', 'ILIKE', '%' . $criteria['university'] . '%');
                });
            }

            if ($criteria['degree']) {
                $studentsQuery->whereHas('educations', function($q) use ($criteria) {
                    $q->where('degree', 'ILIKE', '%' . $criteria['degree'] . '%');
                });
            }

            if ($criteria['field_of_study']) {
                $studentsQuery->whereHas('educations', function($q) use ($criteria) {
                    $q->where('field', 'ILIKE', '%' . $criteria['field_of_study'] . '%');
                });
            }

            // Limit pre-filtering to 100 students for AI ranking
            $students = $studentsQuery->limit(100)->get();

            if ($students->isEmpty()) {
                return response()->json([
                    'success' => true,
                    'data' => [
                        'query' => $query,
                        'extracted_criteria' => $criteria,
                        'results' => [],
                        'total' => 0,
                        'page' => $page,
                        'per_page' => $perPage,
                        'total_pages' => 0,
                    ]
                ]);
            }

            // Step 3: Use AI to rank the pre-filtered students
            $rankings = $this->smartSearchService->rankProfiles($query, $students->all());

            // Step 4: Merge rankings with student data and deduplicate
            $rankedStudents = [];
            $seenIds = []; // Track already processed student IDs
            
            foreach ($rankings as $ranking) {
                $studentId = $ranking['id'];
                
                // Skip if we've already processed this student (keep highest score)
                if (isset($seenIds[$studentId])) {
                    continue;
                }
                
                $student = $students->firstWhere('id', $studentId);
                if ($student) {
                    $seenIds[$studentId] = true;
                    
                    $rankedStudents[] = [
                        'id' => $student->user_id,
                        'score' => $ranking['score'],
                        'reason' => $ranking['reason'],
                        'profile' => [
                            'full_name' => $student->full_name ?? 'N/A',
                            'email' => $student->user->email ?? 'N/A',
                            'phone' => $student->phone,
                            'bio' => $student->summary,
                            'profile_image' => $student->photo_url,
                            'skills' => $student->skills->map(function($skill) {
                                return [
                                    'id' => $skill->id,
                                    'name' => $skill->name,
                                    'proficiency_level' => $skill->pivot->proficiency_level ?? null
                                ];
                            }),
                            'educations' => $student->educations->map(function($edu) {
                                return [
                                    'university' => $edu->school,
                                    'degree_type' => $edu->degree,
                                    'field_of_study' => $edu->field,
                                    'start_date' => $edu->start_year,
                                    'end_date' => $edu->end_year,
                                ];
                            }),
                            'experiences' => $student->experiences->map(function($exp) {
                                return [
                                    'job_title' => $exp->title,
                                    'company_name' => $exp->company,
                                    'description' => $exp->description,
                                    'start_date' => $exp->start_date,
                                    'end_date' => $exp->end_date,
                                ];
                            }),
                        ],
                        'matching_criteria' => $this->getMatchingCriteria($student, $criteria)
                    ];
                }
            }

            // Sort by score descending
            usort($rankedStudents, function($a, $b) {
                return $b['score'] - $a['score'];
            });

            // Paginate results
            $total = count($rankedStudents);
            $totalPages = ceil($total / $perPage);
            $offset = ($page - 1) * $perPage;
            $paginatedResults = array_slice($rankedStudents, $offset, $perPage);

            // Log search activity
            if ($request->user() && $request->user()->role === 'recruiter') {
                ActivityLog::create([
                    'user_id' => $request->user()->id,
                    'action' => 'smart_search',
                    'description' => "Searched: \"$query\"",
                    'metadata' => json_encode([
                        'query' => $query,
                        'criteria' => $criteria,
                        'results_count' => $total
                    ])
                ]);
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'query' => $query,
                    'extracted_criteria' => $criteria,
                    'results' => $paginatedResults,
                    'total' => $total,
                    'page' => $page,
                    'per_page' => $perPage,
                    'total_pages' => $totalPages,
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Smart Search error', [
                'query' => $request->input('query'),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'An error occurred during search',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get search suggestions for autocomplete
     * GET /api/students/search-suggestions
     */
    public function suggestions(Request $request)
    {
        $query = $request->input('q', '');
        
        if (strlen($query) < 2) {
            return response()->json([
                'success' => true,
                'suggestions' => []
            ]);
        }

        // Get suggestions based on existing data
        $suggestions = [];

        // Skill suggestions
        $skills = \DB::table('skills')
            ->where('name', 'ILIKE', '%' . $query . '%')
            ->distinct()
            ->limit(5)
            ->pluck('name');

        foreach ($skills as $skill) {
            $suggestions[] = [
                'type' => 'skill',
                'text' => $skill,
                'query' => "developer with {$skill} skills"
            ];
        }

        // University suggestions
        $universities = \DB::table('educations')
            ->where('university', 'ILIKE', '%' . $query . '%')
            ->distinct()
            ->limit(3)
            ->pluck('university');

        foreach ($universities as $university) {
            $suggestions[] = [
                'type' => 'university',
                'text' => $university,
                'query' => "students from {$university}"
            ];
        }

        return response()->json([
            'success' => true,
            'suggestions' => $suggestions
        ]);
    }

    /**
     * Helper: Determine which criteria matched for a student
     */
    private function getMatchingCriteria($student, $criteria): array
    {
        $matches = [];

        if (!empty($criteria['skills'])) {
            $studentSkills = $student->skills->pluck('name')->map(function($s) {
                return strtolower($s);
            })->toArray();
            
            foreach ($criteria['skills'] as $skill) {
                if (in_array(strtolower($skill), $studentSkills)) {
                    $matches[] = "Skill: {$skill}";
                }
            }
        }

        if ($criteria['university']) {
            foreach ($student->educations as $edu) {
                if (stripos($edu->school, $criteria['university']) !== false) {
                    $matches[] = "University: {$criteria['university']}";
                    break;
                }
            }
        }

        if ($criteria['degree']) {
            foreach ($student->educations as $edu) {
                if (stripos($edu->degree, $criteria['degree']) !== false) {
                    $matches[] = "Degree: {$criteria['degree']}";
                    break;
                }
            }
        }

        if ($criteria['field_of_study']) {
            foreach ($student->educations as $edu) {
                if (stripos($edu->field, $criteria['field_of_study']) !== false) {
                    $matches[] = "Field: {$criteria['field_of_study']}";
                    break;
                }
            }
        }

        return $matches;
    }
}
