<?php

namespace App\Services;

use App\Models\SkillNormalization;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

/**
 * Groq-powered skill normalization service
 * Uses Groq API for intelligent skill categorization
 */
class AISkillNormalizer
{
    /**
     * Normalize a skill using Groq LLM
     * Fast, cost-effective, and good for skill categorization
     */
    public static function normalizeWithGroq(string $rawSkill): ?array
    {
        $apiKey = config('services.groq.key');
        
        if (!$apiKey) {
            Log::warning('GROQ_API_KEY not configured');
            return null;
        }

        // Log presence of API key without revealing it
        Log::info('GROQ_API_KEY present; preparing Groq request');

        try {
            $prompt = self::getSkillNormalizationPrompt($rawSkill);
            Log::info('Calling Groq normalization for skill: ' . $rawSkill);
            
            $endpoint = rtrim(config('services.groq.endpoint', 'https://api.groq.com/openai/v1'), '/');
            $model = config('services.groq.model', 'llama-3.3-70b-versatile');
            $shouldBypassSSL = app()->environment('local') || (bool)config('services.groq.disable_ssl_verify', false);

            $http = Http::timeout(10);
            if ($shouldBypassSSL) {
                $http = $http->withoutVerifying(); // Only in local/dev as configured
            }

            $response = $http
                ->withHeaders([
                    'Content-Type' => 'application/json',
                    'Authorization' => 'Bearer ' . $apiKey,
                ])
                ->post($endpoint . '/chat/completions', [
                    'messages' => [
                        [
                            'role' => 'system',
                            'content' => 'You are a technical skill classifier. Respond ONLY with valid JSON, no markdown or extra text.'
                        ],
                        [
                            'role' => 'user',
                            'content' => $prompt
                        ]
                    ],
                    'model' => $model,
                    'temperature' => 0.1,
                    'max_tokens' => 200,
                ]);

            if ($response->successful()) {
                $data = $response->json();
                $content = $data['choices'][0]['message']['content'] ?? '';
                
                // Clean up response (remove markdown code blocks if present)
                $content = preg_replace('/```json\n?|\n?```/', '', $content);
                $content = trim($content);
                
                $result = json_decode($content, true);
                
                if ($result && isset($result['canonical_name'])) {
                    Log::info("AI normalized skill: $rawSkill -> " . $result['canonical_name']);
                    self::storeNormalization($rawSkill, $result);
                    return $result;
                }
            } else {
                Log::warning('Groq API call failed', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Groq API error: ' . $e->getMessage());
        }

        return null;
    }

    /**
     * Get normalization prompt for LLM
     */
    private static function getSkillNormalizationPrompt(string $rawSkill): string
    {
        return <<<PROMPT
Normalize this technical skill to a canonical category. Respond with ONLY valid JSON (no markdown formatting).

Raw skill: "$rawSkill"

Respond with exactly this JSON format (and nothing else):
{
  "canonical_name": "JavaScript",
  "category": "Language",
  "confidence": 0.95,
  "reason": "Exact match",
  "aliases": ["js", "java script", "es6"]
}

Categories: Language, Framework, Runtime, Database, Cloud, Tool, OS, Soft Skill

Be specific with your canonical name. Examples:
- "js" or "java script" -> "JavaScript"
- "node" or "nodejs" -> "Node.js"
- "react" -> "React"
- "sql" or "mysql" -> "SQL"
- "python" -> "Python"
- "cloud" or "aws" -> "AWS"

JSON Response:
PROMPT;
    }

    /**
     * Persist AI-normalized skill so we avoid repeat API calls next time.
     */
    private static function storeNormalization(string $rawSkill, array $result): void
    {
        try {
            $canonical = $result['canonical_name'] ?? null;
            if (!$canonical) {
                return;
            }

            $aliases = $result['aliases'] ?? [];
            if (!is_array($aliases)) {
                $aliases = [];
            }

            // Always include the raw skill text as an alias for lookup.
            $aliases[] = $rawSkill;

            $existing = SkillNormalization::where('canonical_name', $canonical)->first();

            if ($existing) {
                $merged = array_values(array_unique(array_merge($existing->aliases ?? [], $aliases)));
                $existing->aliases = $merged;
                // Update category if AI provided one and we don't have it yet.
                if (!$existing->category && !empty($result['category'])) {
                    $existing->category = $result['category'];
                }
                $existing->save();
            } else {
                SkillNormalization::create([
                    'canonical_name' => $canonical,
                    'aliases' => array_values(array_unique($aliases)),
                    'category' => $result['category'] ?? null,
                ]);
            }
        } catch (\Throwable $e) {
            Log::warning('Failed to store AI skill normalization', [
                'error' => $e->getMessage(),
            ]);
        }
    }

}
