<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use App\Models\User;
use App\Models\Student;
use App\Models\Skill;
use App\Models\Education;
use App\Models\Experience;

class TestStudentsSeeder extends Seeder
{
    public function run()
    {
        $students = [
            [
                'name' => 'Ahmed Bennani',
                'email' => 'ahmed.bennani@test.com',
                'bio' => 'Passionate full-stack developer with expertise in modern web technologies',
                'phone' => '+212661234567',
                'skills' => ['JavaScript', 'React', 'Node.js', 'MongoDB', 'TypeScript', 'Express.js'],
                'education' => ['school' => 'ENSAM Casablanca', 'degree' => 'Master', 'field' => 'Software Engineering'],
                'experiences' => [
                    ['title' => 'Full Stack Developer', 'company' => 'TechCorp Morocco', 'start_date' => '2022-01-15', 'end_date' => null]
                ]
            ],
            [
                'name' => 'Fatima Zahra El Amrani',
                'email' => 'fatima.elamrani@test.com',
                'bio' => 'Data scientist specializing in machine learning and AI',
                'phone' => '+212662345678',
                'skills' => ['Python', 'Machine Learning', 'TensorFlow', 'Pandas', 'NumPy', 'Scikit-learn'],
                'education' => ['school' => 'Université Mohammed V', 'degree' => 'PhD', 'field' => 'Data Science'],
                'experiences' => [
                    ['title' => 'Data Scientist', 'company' => 'AI Solutions Ltd', 'start_date' => '2020-06-01', 'end_date' => null]
                ]
            ],
            [
                'name' => 'Youssef Alaoui',
                'email' => 'youssef.alaoui@test.com',
                'bio' => 'Frontend engineer focused on creating beautiful user experiences',
                'phone' => '+212663456789',
                'skills' => ['React', 'Vue.js', 'CSS', 'HTML', 'Tailwind CSS', 'Figma'],
                'education' => ['school' => 'ENSAM Rabat', 'degree' => 'Bachelor', 'field' => 'Computer Science'],
                'experiences' => [
                    ['title' => 'Frontend Developer', 'company' => 'WebDesign Pro', 'start_date' => '2021-03-20', 'end_date' => '2023-12-31']
                ]
            ],
            [
                'name' => 'Salma Idrissi',
                'email' => 'salma.idrissi@test.com',
                'bio' => 'Backend developer with strong database design skills',
                'phone' => '+212664567890',
                'skills' => ['Java', 'Spring Boot', 'PostgreSQL', 'MySQL', 'Docker', 'Kubernetes'],
                'education' => ['school' => 'INPT Rabat', 'degree' => 'Master', 'field' => 'Software Engineering'],
                'experiences' => [
                    ['title' => 'Backend Engineer', 'company' => 'CloudTech Inc', 'start_date' => '2019-09-01', 'end_date' => null]
                ]
            ],
            [
                'name' => 'Mehdi Tazi',
                'email' => 'mehdi.tazi@test.com',
                'bio' => 'Mobile app developer for iOS and Android platforms',
                'phone' => '+212665678901',
                'skills' => ['React Native', 'Flutter', 'Swift', 'Kotlin', 'Firebase', 'Redux'],
                'education' => ['school' => 'ENSIAS Rabat', 'degree' => 'Master', 'field' => 'Mobile Computing'],
                'experiences' => [
                    ['title' => 'Mobile Developer', 'company' => 'AppMakers Studio', 'start_date' => '2021-07-15', 'end_date' => null]
                ]
            ],
            [
                'name' => 'Nadia Belkacem',
                'email' => 'nadia.belkacem@test.com',
                'bio' => 'DevOps engineer passionate about automation and cloud infrastructure',
                'phone' => '+212666789012',
                'skills' => ['AWS', 'Azure', 'Docker', 'Kubernetes', 'Jenkins', 'Terraform'],
                'education' => ['school' => 'ENSAM Meknes', 'degree' => 'Master', 'field' => 'Cloud Computing'],
                'experiences' => [
                    ['title' => 'DevOps Engineer', 'company' => 'Infrastructure Co', 'start_date' => '2020-02-10', 'end_date' => null]
                ]
            ],
            [
                'name' => 'Omar Chakir',
                'email' => 'omar.chakir@test.com',
                'bio' => 'Cybersecurity specialist focused on network security',
                'phone' => '+212667890123',
                'skills' => ['Network Security', 'Penetration Testing', 'Python', 'Linux', 'Wireshark', 'Metasploit'],
                'education' => ['school' => 'EMI Rabat', 'degree' => 'Master', 'field' => 'Cybersecurity'],
                'experiences' => [
                    ['title' => 'Security Analyst', 'company' => 'SecureNet Systems', 'start_date' => '2021-01-05', 'end_date' => null]
                ]
            ],
            [
                'name' => 'Khadija Mansouri',
                'email' => 'khadija.mansouri@test.com',
                'bio' => 'UI/UX designer with a passion for user-centered design',
                'phone' => '+212668901234',
                'skills' => ['Figma', 'Adobe XD', 'Sketch', 'Photoshop', 'User Research', 'Prototyping'],
                'education' => ['school' => 'ENSAD Casablanca', 'degree' => 'Bachelor', 'field' => 'Design'],
                'experiences' => [
                    ['title' => 'UI/UX Designer', 'company' => 'Creative Agency', 'start_date' => '2022-03-01', 'end_date' => null]
                ]
            ],
            [
                'name' => 'Amine Berrada',
                'email' => 'amine.berrada@test.com',
                'bio' => 'Full-stack developer specializing in Python and Django',
                'phone' => '+212669012345',
                'skills' => ['Python', 'Django', 'REST API', 'PostgreSQL', 'Redis', 'Celery'],
                'education' => ['school' => 'MIT', 'degree' => 'Master', 'field' => 'Computer Science'],
                'experiences' => [
                    ['title' => 'Python Developer', 'company' => 'StartupHub', 'start_date' => '2020-05-20', 'end_date' => null]
                ]
            ],
            [
                'name' => 'Sophia Chen',
                'email' => 'sophia.chen@test.com',
                'bio' => 'AI researcher working on natural language processing',
                'phone' => '+212660123456',
                'skills' => ['Python', 'NLP', 'Deep Learning', 'PyTorch', 'BERT', 'Transformers'],
                'education' => ['school' => 'Stanford University', 'degree' => 'PhD', 'field' => 'Artificial Intelligence'],
                'experiences' => [
                    ['title' => 'AI Researcher', 'company' => 'AI Labs', 'start_date' => '2019-08-01', 'end_date' => null]
                ]
            ],
            [
                'name' => 'Karim Essaidi',
                'email' => 'karim.essaidi@test.com',
                'bio' => 'Junior developer eager to learn and grow in web development',
                'phone' => '+212661234560',
                'skills' => ['HTML', 'CSS', 'JavaScript', 'Git', 'React'],
                'education' => ['school' => 'ENSAM Casablanca', 'degree' => 'Bachelor', 'field' => 'Computer Science'],
                'experiences' => []
            ],
            [
                'name' => 'Leila Chraibi',
                'email' => 'leila.chraibi@test.com',
                'bio' => 'Blockchain developer interested in DeFi applications',
                'phone' => '+212662345671',
                'skills' => ['Solidity', 'Ethereum', 'Web3.js', 'Smart Contracts', 'Blockchain', 'JavaScript'],
                'education' => ['school' => 'INPT Rabat', 'degree' => 'Master', 'field' => 'Blockchain Technology'],
                'experiences' => [
                    ['title' => 'Blockchain Developer', 'company' => 'CryptoSolutions', 'start_date' => '2021-10-01', 'end_date' => null]
                ]
            ],
            [
                'name' => 'Hassan Benjelloun',
                'email' => 'hassan.benjelloun@test.com',
                'bio' => 'Game developer passionate about creating immersive experiences',
                'phone' => '+212663456782',
                'skills' => ['Unity', 'C#', 'Unreal Engine', 'Blender', '3D Modeling', 'Game Design'],
                'education' => ['school' => 'ENSAM Meknes', 'degree' => 'Bachelor', 'field' => 'Game Development'],
                'experiences' => [
                    ['title' => 'Game Developer', 'company' => 'GameStudio Pro', 'start_date' => '2022-06-15', 'end_date' => null]
                ]
            ],
            [
                'name' => 'Imane Lahlou',
                'email' => 'imane.lahlou@test.com',
                'bio' => 'Data engineer building scalable data pipelines',
                'phone' => '+212664567893',
                'skills' => ['Python', 'Apache Spark', 'Hadoop', 'Airflow', 'SQL', 'ETL'],
                'education' => ['school' => 'Université Mohammed V', 'degree' => 'Master', 'field' => 'Data Engineering'],
                'experiences' => [
                    ['title' => 'Data Engineer', 'company' => 'BigData Corp', 'start_date' => '2020-11-01', 'end_date' => null]
                ]
            ],
            [
                'name' => 'Rachid El Fassi',
                'email' => 'rachid.elfassi@test.com',
                'bio' => 'Senior software architect with 8 years of experience',
                'phone' => '+212665678904',
                'skills' => ['Java', 'Microservices', 'System Design', 'Cloud Architecture', 'Docker', 'Kubernetes'],
                'education' => ['school' => 'ENSIAS Rabat', 'degree' => 'Master', 'field' => 'Software Engineering'],
                'experiences' => [
                    ['title' => 'Software Architect', 'company' => 'Enterprise Solutions', 'start_date' => '2016-03-01', 'end_date' => null]
                ]
            ],
        ];

        foreach ($students as $studentData) {
            // Check if user already exists
            if (User::where('email', $studentData['email'])->exists()) {
                $this->command->warn("Skipping {$studentData['name']} - email already exists");
                continue;
            }

            // Create user
            $user = User::create([
                'id' => (string) Str::uuid(),
                'email' => $studentData['email'],
                'email_verified_at' => now(),
                'password' => Hash::make('password123'),
                'role' => 'student',
            ]);

            // Create student profile
            $student = Student::create([
                'user_id' => $user->id,
                'full_name' => $studentData['name'],
                'phone' => $studentData['phone'],
                'summary' => $studentData['bio'],
                'is_card_active' => true,
            ]);

            // Add skills
            foreach ($studentData['skills'] as $skillName) {
                $skill = Skill::firstOrCreate(
                    ['student_id' => $student->user_id, 'name' => $skillName],
                    [
                        'id' => (string) Str::uuid(),
                        'level' => ['Beginner', 'Intermediate', 'Advanced', 'Expert'][rand(1, 3)],
                    ]
                );
            }

            // Add education
            $edu = $studentData['education'];
            Education::create([
                'id' => (string) Str::uuid(),
                'student_id' => $student->user_id,
                'school' => $edu['school'],
                'degree' => $edu['degree'],
                'field' => $edu['field'],
                'start_year' => (string) (date('Y') - rand(2, 6)),
                'end_year' => (string) (date('Y') - rand(0, 1)),
                'description' => 'Focused on modern software development practices and technologies.',
            ]);

            // Add experiences
            foreach ($studentData['experiences'] as $expData) {
                Experience::create([
                    'id' => (string) Str::uuid(),
                    'student_id' => $student->user_id,
                    'title' => $expData['title'],
                    'company' => $expData['company'],
                    'start_date' => $expData['start_date'],
                    'end_date' => $expData['end_date'],
                    'description' => 'Contributed to various projects and collaborated with cross-functional teams.',
                ]);
            }

            $this->command->info("Created student: {$studentData['name']}");
        }

        $this->command->info("\n✅ Successfully created 15 test students with skills, education, and experiences!");
        $this->command->info("📧 All accounts use password: password123");
    }
}
