<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Laravel\Sanctum\Sanctum;
use Tests\TestCase;
use App\Models\User;
use App\Models\Recruiter;
use App\Models\Student;
use App\Models\Shortlist;

class ShortlistTest extends TestCase
{
    use RefreshDatabase;

    public function setUp(): void
    {
        parent::setUp();
        // Create recruiter and student users
        $this->recruiterUser = User::factory()->create(['role' => 'recruiter']);
        $this->recruiter = Recruiter::factory()->create(['user_id' => $this->recruiterUser->id]);
        $this->studentUser = User::factory()->create(['role' => 'student']);
        $this->student = Student::factory()->create(['user_id' => $this->studentUser->id]);
    }

    public function test_recruiter_can_add_student_to_shortlist()
    {
        Sanctum::actingAs($this->recruiterUser, ['*']);
        $response = $this->postJson('/api/shortlist', [
            'student_id' => $this->student->user_id,
            'private_note' => 'Promising candidate',
        ]);
        $response->assertStatus(201)
            ->assertJson(['message' => 'Student added to shortlist']);
        $this->assertDatabaseHas('shortlists', [
            'recruiter_id' => $this->recruiterUser->id,
            'student_id' => $this->student->user_id,
            'private_note' => 'Promising candidate',
        ]);
    }

    public function test_cannot_add_same_student_twice()
    {
        Sanctum::actingAs($this->recruiterUser, ['*']);
        Shortlist::create([
            'recruiter_id' => $this->recruiterUser->id,
            'student_id' => $this->student->user_id,
        ]);
        $response = $this->postJson('/api/shortlist', [
            'student_id' => $this->student->user_id,
        ]);
        $response->assertStatus(409)
            ->assertJson(['message' => 'Student already in shortlist']);
    }

    public function test_note_is_optional()
    {
        Sanctum::actingAs($this->recruiterUser, ['*']);
        $response = $this->postJson('/api/shortlist', [
            'student_id' => $this->student->user_id,
        ]);
        $response->assertStatus(201);
        $this->assertDatabaseHas('shortlists', [
            'recruiter_id' => $this->recruiterUser->id,
            'student_id' => $this->student->user_id,
            'private_note' => null,
        ]);
    }
} 