import React, { useState, useEffect, useCallback } from 'react';
import { useNavigate } from 'react-router-dom';
import apiClient from '../apiClient';
import ProfileRecruiter from './ProfileRecruiter';
import { 
  FiUser, 
  FiMail, 
  FiPhone, 
  FiLinkedin, 
  FiGithub, 
  FiBook, 
  FiBriefcase, 
  FiSave, 
  FiLoader, 
  FiArrowLeft, 
  FiPlus, 
  FiTrash2, 
  FiEdit3, 
  FiSettings, 
  FiMapPin,
  FiCalendar,
  FiTarget,
  FiStar,
  FiCheck,
  FiAlertCircle,
  FiX,
  FiUpload,
  FiInfo
} from 'react-icons/fi';

interface Experience {
  id?: string;
  company: string;
  title: string;
  start_date: string;
  end_date?: string;
  description?: string;
}

interface Skill {
  id?: string;
  name: string;
  level?: string;
}

interface Education {
  id?: string;
  school: string;
  degree?: string;
  field?: string;
  start_year?: string;
  end_year?: string;
  description?: string;
}

interface User {
  id: string;
  email: string;
  role: string;
  profile?: {
    full_name?: string;
    university?: string;
    phone?: string;
    linkedin?: string;
    github?: string;
    summary?: string;
    languages?: string[] | string;
    company_name?: string;
    photo_url?: string;
    photo?: string;
    graduation_year?: string;
    degree?: string;
  };
}

const Profile: React.FC = () => {
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [errors, setErrors] = useState<{ [key: string]: string }>({});
  const [success, setSuccess] = useState('');
  const [activeTab, setActiveTab] = useState('personal');
  const navigate = useNavigate();
  const [photoUploading, setPhotoUploading] = useState(false);
  const [photoUrl, setPhotoUrl] = useState<string | null>(null);

  // Experience and Skills state (only for students)
  const [experiences, setExperiences] = useState<Experience[]>([]);
  const [skills, setSkills] = useState<Skill[]>([]);
  const [educations, setEducations] = useState<Education[]>([]);
  const [experiencesLoading, setExperiencesLoading] = useState(false);
  const [skillsLoading, setSkillsLoading] = useState(false);
  const [educationsLoading, setEducationsLoading] = useState(false);

  // AI Tips state
  const [aiTips, setAiTips] = useState<any[]>([]);
  const [tipsLoading, setTipsLoading] = useState(false);

  // Form modals state
  const [showExperienceModal, setShowExperienceModal] = useState(false);
  const [showSkillModal, setShowSkillModal] = useState(false);
  const [showEducationModal, setShowEducationModal] = useState(false);
  const [editingExperience, setEditingExperience] = useState<Experience | null>(null);
  const [editingSkill, setEditingSkill] = useState<Skill | null>(null);
  const [editingEducation, setEditingEducation] = useState<Education | null>(null);

  const [formData, setFormData] = useState({
    fullName: '',
    email: '',
    phone: '',
    linkedin: '',
    github: '',
    summary: '',
    university: '',
    graduationYear: '',
    languages: '',
    companyName: '',
  });

  // Form data for modals
  const [experienceForm, setExperienceForm] = useState<Experience>({
    company: '',
    title: '',
    start_date: '',
    end_date: '',
    description: '',
  });

  const [skillForm, setSkillForm] = useState<Skill>({
    name: '',
    level: '',
  });

  const [educationForm, setEducationForm] = useState<Education>({
    school: '',
    degree: '',
    field: '',
    start_year: '',
    end_year: '',
    description: '',
  });

  const fetchUserData = useCallback(async () => {
    try {
      const response = await apiClient.get('/api/user');
      const userData = response.data.data?.user || response.data.user;
      setUser(userData);
      
      // If recruiter, stop here and let ProfileRecruiter handle it
      if (userData.role === 'recruiter') {
        setLoading(false);
        return;
      }
      
      console.log('User data:', userData);
      console.log('Profile:', userData.profile);
      console.log('Photo URL from profile:', userData.profile?.photo_url);
      console.log('Photo from profile:', userData.profile?.photo);
      
      // Populate form data
      setFormData({
        fullName: userData.profile?.full_name || '',
        email: userData.email || '',
        phone: userData.profile?.phone || '',
        linkedin: userData.profile?.linkedin || '',
        github: userData.profile?.github || '',
        summary: userData.profile?.summary || '',
        university: userData.profile?.university || '',
        graduationYear: userData.profile?.graduation_year || '',
        languages: Array.isArray(userData.profile?.languages) 
          ? userData.profile.languages.join(', ') 
          : userData.profile?.languages || '',
        companyName: userData.profile?.company_name || '',
      });
      const photoUrlFromProfile = userData.profile?.photo_url || userData.profile?.photo || null;
      console.log('Setting photo URL to:', photoUrlFromProfile);
      setPhotoUrl(photoUrlFromProfile);
    } catch (error) {
      console.error('Error fetching user data:', error);
      navigate('/login');
    } finally {
      setLoading(false);
    }
  }, [navigate]);

  // Fetch functions for experiences, educations, and skills
  const fetchExperiences = useCallback(async () => {
    if (!user?.id) return;
    setExperiencesLoading(true);
    try {
      const response = await apiClient.get(`/api/students/${user.id}/experiences`);
      setExperiences(response.data.experiences || []);
    } catch (error) {
      console.error('Error fetching experiences:', error);
    } finally {
      setExperiencesLoading(false);
    }
  }, [user?.id]);

  const fetchSkills = useCallback(async () => {
    if (!user?.id) return;
    setSkillsLoading(true);
    try {
      const response = await apiClient.get(`/api/students/${user.id}/skills`);
      setSkills(response.data.skills || []);
    } catch (error) {
      console.error('Error fetching skills:', error);
    } finally {
      setSkillsLoading(false);
    }
  }, [user?.id]);

  const fetchEducations = useCallback(async () => {
    if (!user?.id) return;
    setEducationsLoading(true);
    try {
      const response = await apiClient.get(`/api/students/${user.id}/educations`);
      setEducations(response.data.educations || []);
    } catch (error) {
      console.error('Error fetching educations:', error);
    } finally {
      setEducationsLoading(false);
    }
  }, [user?.id]);

  const fetchAiTips = useCallback(async () => {
    setTipsLoading(true);
    try {
      const response = await apiClient.get('/api/student/pro-tips');
      const tipsData = response.data?.tips || [];
      setAiTips(tipsData);
    } catch (error) {
      console.error('Error fetching AI tips:', error);
      setAiTips([]);
    } finally {
      setTipsLoading(false);
    }
  }, []);

  // CRUD operations for experiences
  const handleSaveExperience = async () => {
    if (!user?.id) return;
    
    try {
      if (editingExperience?.id) {
        // Update existing
        await apiClient.put(`/api/experiences/${editingExperience.id}`, experienceForm);
      } else {
        // Create new
        await apiClient.post(`/api/students/${user.id}/experiences`, experienceForm);
      }
      
      setShowExperienceModal(false);
      setEditingExperience(null);
      setExperienceForm({ company: '', title: '', start_date: '', end_date: '', description: '' });
      fetchExperiences();
      setSuccess('Experience saved successfully!');
    } catch (error) {
      console.error('Error saving experience:', error);
      setErrors({ experience: 'Failed to save experience' });
    }
  };

  const handleDeleteExperience = async (id: string) => {
    if (!window.confirm('Are you sure you want to delete this experience?')) return;
    
    try {
      await apiClient.delete(`/api/experiences/${id}`);
      fetchExperiences();
      setSuccess('Experience deleted successfully!');
    } catch (error) {
      console.error('Error deleting experience:', error);
      setErrors({ experience: 'Failed to delete experience' });
    }
  };

  // CRUD operations for skills
  const handleSaveSkill = async () => {
    if (!user?.id) return;
    
    try {
      if (editingSkill?.id) {
        // Update existing
        await apiClient.put(`/api/skills/${editingSkill.id}`, skillForm);
      } else {
        // Create new
        await apiClient.post(`/api/students/${user.id}/skills`, skillForm);
      }
      
      setShowSkillModal(false);
      setEditingSkill(null);
      setSkillForm({ name: '', level: '' });
      fetchSkills();
      setSuccess('Skill saved successfully!');
    } catch (error) {
      console.error('Error saving skill:', error);
      setErrors({ skill: 'Failed to save skill' });
    }
  };

  const handleDeleteSkill = async (id: string) => {
    if (!window.confirm('Are you sure you want to delete this skill?')) return;
    
    try {
      await apiClient.delete(`/api/skills/${id}`);
      fetchSkills();
      setSuccess('Skill deleted successfully!');
    } catch (error) {
      console.error('Error deleting skill:', error);
      setErrors({ skill: 'Failed to delete skill' });
    }
  };

  // CRUD operations for educations
  const handleSaveEducation = async () => {
    if (!user?.id) return;
    
    try {
      if (editingEducation?.id) {
        // Update existing
        await apiClient.put(`/api/educations/${editingEducation.id}`, educationForm);
      } else {
        // Create new
        await apiClient.post(`/api/students/${user.id}/educations`, educationForm);
      }
      
      setShowEducationModal(false);
      setEditingEducation(null);
      setEducationForm({ school: '', degree: '', field: '', start_year: '', end_year: '', description: '' });
      fetchEducations();
      setSuccess('Education saved successfully!');
    } catch (error) {
      console.error('Error saving education:', error);
      setErrors({ education: 'Failed to save education' });
    }
  };

  const handleDeleteEducation = async (id: string) => {
    if (!window.confirm('Are you sure you want to delete this education?')) return;
    
    try {
      await apiClient.delete(`/api/educations/${id}`);
      fetchEducations();
      setSuccess('Education deleted successfully!');
    } catch (error) {
      console.error('Error deleting education:', error);
      setErrors({ education: 'Failed to delete education' });
    }
  };

  useEffect(() => {
    const token = localStorage.getItem('token');
    if (!token) {
      navigate('/login');
      return;
    }

    fetchUserData();
  }, [navigate, fetchUserData]);

  // Fetch experiences and skills when user data is loaded
  useEffect(() => {
    if (user?.id && user.role === 'student') {
      fetchExperiences();
      fetchSkills();
      fetchEducations();
      fetchAiTips();
    }
  }, [user?.id, user?.role, fetchExperiences, fetchSkills, fetchEducations, fetchAiTips]);

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
    if (errors[name]) {
      setErrors(prev => ({ ...prev, [name]: '' }));
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setSaving(true);
    setErrors({});
    setSuccess('');

    try {
      const updateData: Record<string, any> = {};

      // Only include fields that have values (optional fields)
      if (formData.fullName?.trim()) updateData.full_name = formData.fullName;
      if (formData.phone?.trim()) updateData.phone = formData.phone;
      if (formData.summary?.trim()) updateData.summary = formData.summary;
      if (formData.university?.trim()) updateData.university = formData.university;
      if (formData.graduationYear?.trim()) updateData.graduation_year = formData.graduationYear;
      if (formData.linkedin?.trim()) updateData.linkedin = formData.linkedin;
      if (formData.github?.trim()) updateData.github = formData.github;
      if (formData.languages?.trim()) {
        updateData.languages = formData.languages.split(',').map((lang: string) => lang.trim());
      }
      if (user?.role === 'recruiter' && formData.companyName?.trim()) {
        updateData.company_name = formData.companyName;
      }

      await apiClient.put('/api/profile', updateData);
      setSuccess('Profile updated successfully!');
      fetchUserData(); // Refresh user data
    } catch (error: unknown) {
      const err = error as { response?: { data?: { errors?: Record<string, string[]>; message?: string } } };
      if (err.response?.data?.errors) {
        const newErrors: { [key: string]: string } = {};
        Object.entries(err.response.data.errors).forEach(([key, messages]) => {
          newErrors[key] = messages[0];
        });
        setErrors(newErrors);
      } else {
        setErrors({ general: err.response?.data?.message || 'An error occurred' });
      }
    } finally {
      setSaving(false);
    }
  };

  const handlePhotoUpload = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;

    if (file.size > 5 * 1024 * 1024) {
      setErrors({ photo: 'Photo must be less than 5MB' });
      return;
    }

    const formData = new FormData();
    formData.append('photo', file);

    setPhotoUploading(true);
    try {
      const response = await apiClient.post('/api/profile/photo', formData, {
        headers: { 'Content-Type': 'multipart/form-data' },
      });
      console.log('Photo upload response:', response.data);
      const newPhotoUrl = response.data.photo_url;
      console.log('Setting photo URL to:', newPhotoUrl);
      setPhotoUrl(newPhotoUrl);
      
      // Update user state with new photo URL
      if (user) {
        setUser({
          ...user,
          profile: {
            ...user.profile,
            photo_url: newPhotoUrl,
          }
        });
      }
      
      setSuccess('Photo uploaded successfully!');
    } catch (error) {
      console.error('Error uploading photo:', error);
      setErrors({ photo: 'Failed to upload photo' });
    } finally {
      setPhotoUploading(false);
    }
  };

  // If recruiter, render the recruiter-specific profile component
  if (user?.role === 'recruiter') {
    return <ProfileRecruiter />;
  }

  if (loading) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-100 flex items-center justify-center">
        <div className="text-center">
          <div className="relative">
            <div className="w-16 h-16 bg-gradient-to-r from-blue-600 via-purple-600 to-indigo-600 rounded-2xl flex items-center justify-center shadow-xl mx-auto mb-6">
              <FiLoader className="animate-spin text-white text-2xl" />
            </div>
            <div className="absolute inset-0 bg-gradient-to-r from-blue-600 via-purple-600 to-indigo-600 rounded-2xl blur-lg opacity-30 animate-pulse"></div>
          </div>
          <h3 className="text-xl font-bold text-gray-900 mb-2">Loading Your Profile</h3>
          <p className="text-gray-600">Preparing your professional dashboard...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-100 py-12">
      <div className="max-w-7xl mx-auto px-4">
        {/* Ultra Modern Header */}
        <div className="text-center mb-12">
          <div className="flex items-center justify-center mb-6">
            <div className="relative">
              <div className="w-16 h-16 bg-gradient-to-r from-blue-600 via-purple-600 to-indigo-600 rounded-2xl flex items-center justify-center shadow-xl">
                <FiUser className="text-white text-2xl" />
              </div>
              <div className="absolute -inset-1 bg-gradient-to-r from-blue-600 via-purple-600 to-indigo-600 rounded-2xl blur-lg opacity-30 animate-pulse"></div>
            </div>
          </div>
          <div className="flex items-center justify-center space-x-4 mb-6">
            <button
              onClick={() => navigate(-1)}
              className="group flex items-center space-x-2 text-gray-600 hover:text-blue-600 transition-colors duration-300"
            >
              <div className="w-8 h-8 bg-gray-100 group-hover:bg-blue-50 rounded-lg flex items-center justify-center transition-colors duration-300">
                <FiArrowLeft className="text-sm" />
              </div>
              <span className="font-medium">Back</span>
            </button>
            <div className="w-px h-6 bg-gray-300"></div>
            <h1 className="text-5xl font-bold">
              <span className="bg-gradient-to-r from-blue-600 via-purple-600 to-indigo-600 bg-clip-text text-transparent">
                Profile Settings
              </span>
            </h1>
          </div>
          <p className="text-xl text-gray-600 max-w-3xl mx-auto leading-relaxed">
            Manage your professional information and enhance your career presence.
            <br />
            <span className="text-gray-500">Keep your profile updated to attract the best opportunities.</span>
          </p>
        </div>

        {/* Advanced Profile Completion Tracker */}
        <div className="max-w-4xl mx-auto mb-12">
          <div className="relative overflow-hidden bg-white/80 backdrop-blur-xl border border-white/40 rounded-3xl shadow-2xl p-8">
            <div className="absolute inset-0 bg-gradient-to-br from-blue-500/5 via-purple-500/5 to-indigo-500/5"></div>
            <div className="relative z-10">
              <div className="flex items-center justify-between mb-6">
                <div className="flex items-center space-x-4">
                  <div className="w-14 h-14 bg-gradient-to-r from-blue-600 to-purple-600 rounded-xl flex items-center justify-center shadow-lg">
                    <FiTarget className="text-white text-xl" />
                  </div>
                  <div>
                    <h3 className="text-2xl font-bold text-gray-900">Profile Completion</h3>
                    <p className="text-gray-600">Complete your profile to increase your visibility</p>
                  </div>
                </div>
                <div className="text-right">
                  <div className="text-3xl font-bold bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent">
                    {Math.round(((user?.profile?.full_name ? 1 : 0) + 
                                (user?.profile?.phone ? 1 : 0) + 
                                (user?.profile?.summary ? 1 : 0) + 
                                (user?.profile?.linkedin ? 1 : 0) + 
                                (experiences.length > 0 ? 1 : 0) + 
                                (skills.length > 0 ? 1 : 0) + 
                                (photoUrl ? 1 : 0)) / 7 * 100)}%
                  </div>
                  <p className="text-sm text-gray-500">Complete</p>
                </div>
              </div>
              
              {/* Enhanced Progress Bar */}
              <div className="mb-6">
                <div className="w-full bg-gray-200 rounded-full h-3 overflow-hidden">
                  <div 
                    className="h-full bg-gradient-to-r from-blue-500 via-purple-500 to-indigo-500 rounded-full transition-all duration-1000 ease-out relative"
                    style={{ 
                      width: `${Math.round(((user?.profile?.full_name ? 1 : 0) + 
                                          (user?.profile?.phone ? 1 : 0) + 
                                          (user?.profile?.summary ? 1 : 0) + 
                                          (user?.profile?.linkedin ? 1 : 0) + 
                                          (experiences.length > 0 ? 1 : 0) + 
                                          (skills.length > 0 ? 1 : 0) + 
                                          (photoUrl ? 1 : 0)) / 7 * 100)}%` 
                    }}
                  >
                    <div className="absolute inset-0 bg-white/30 animate-pulse"></div>
                  </div>
                </div>
              </div>

              {/* Interactive Completion Checklist */}
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                <div className={`p-4 rounded-xl border-2 transition-all duration-300 ${
                  user?.profile?.full_name 
                    ? 'bg-emerald-50 border-emerald-200 hover:bg-emerald-100' 
                    : 'bg-gray-50 border-gray-200 hover:bg-yellow-50 hover:border-yellow-300'
                }`}>
                  <div className="flex items-center space-x-3">
                    <div className={`w-8 h-8 rounded-lg flex items-center justify-center ${
                      user?.profile?.full_name ? 'bg-emerald-500' : 'bg-gray-300'
                    }`}>
                      {user?.profile?.full_name ? (
                        <FiCheck className="text-white text-sm" />
                      ) : (
                        <FiUser className="text-gray-600 text-sm" />
                      )}
                    </div>
                    <div>
                      <p className="font-semibold text-gray-900 text-sm">Basic Info</p>
                      <p className="text-xs text-gray-600">Name & contact</p>
                    </div>
                  </div>
                </div>

                <div className={`p-4 rounded-xl border-2 transition-all duration-300 ${
                  photoUrl 
                    ? 'bg-emerald-50 border-emerald-200 hover:bg-emerald-100' 
                    : 'bg-gray-50 border-gray-200 hover:bg-yellow-50 hover:border-yellow-300'
                }`}>
                  <div className="flex items-center space-x-3">
                    <div className={`w-8 h-8 rounded-lg flex items-center justify-center ${
                      photoUrl ? 'bg-emerald-500' : 'bg-gray-300'
                    }`}>
                      {photoUrl ? (
                        <FiCheck className="text-white text-sm" />
                      ) : (
                        <FiUser className="text-gray-600 text-sm" />
                      )}
                    </div>
                    <div>
                      <p className="font-semibold text-gray-900 text-sm">Photo</p>
                      <p className="text-xs text-gray-600">Profile picture</p>
                    </div>
                  </div>
                </div>

                <div className={`p-4 rounded-xl border-2 transition-all duration-300 ${
                  experiences.length > 0 
                    ? 'bg-emerald-50 border-emerald-200 hover:bg-emerald-100' 
                    : 'bg-gray-50 border-gray-200 hover:bg-yellow-50 hover:border-yellow-300'
                }`}>
                  <div className="flex items-center space-x-3">
                    <div className={`w-8 h-8 rounded-lg flex items-center justify-center ${
                      experiences.length > 0 ? 'bg-emerald-500' : 'bg-gray-300'
                    }`}>
                      {experiences.length > 0 ? (
                        <FiCheck className="text-white text-sm" />
                      ) : (
                        <FiBriefcase className="text-gray-600 text-sm" />
                      )}
                    </div>
                    <div>
                      <p className="font-semibold text-gray-900 text-sm">Experience</p>
                      <p className="text-xs text-gray-600">{experiences.length} items</p>
                    </div>
                  </div>
                </div>

                <div className={`p-4 rounded-xl border-2 transition-all duration-300 ${
                  skills.length > 0 
                    ? 'bg-emerald-50 border-emerald-200 hover:bg-emerald-100' 
                    : 'bg-gray-50 border-gray-200 hover:bg-yellow-50 hover:border-yellow-300'
                }`}>
                  <div className="flex items-center space-x-3">
                    <div className={`w-8 h-8 rounded-lg flex items-center justify-center ${
                      skills.length > 0 ? 'bg-emerald-500' : 'bg-gray-300'
                    }`}>
                      {skills.length > 0 ? (
                        <FiCheck className="text-white text-sm" />
                      ) : (
                        <FiSettings className="text-gray-600 text-sm" />
                      )}
                    </div>
                    <div>
                      <p className="font-semibold text-gray-900 text-sm">Skills</p>
                      <p className="text-xs text-gray-600">{skills.length} items</p>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* Enhanced Profile Statistics */}
        <div className="max-w-4xl mx-auto mb-12">
          <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
            <div className="group relative overflow-hidden bg-white/80 backdrop-blur-xl border border-white/40 rounded-2xl shadow-xl p-6 hover:shadow-2xl transition-all duration-500 hover:-translate-y-2">
              <div className="absolute inset-0 bg-gradient-to-br from-blue-500/10 to-purple-500/10 opacity-0 group-hover:opacity-100 transition-opacity duration-500"></div>
              <div className="relative z-10">
                <div className="flex items-center justify-between mb-3">
                  <div className="w-12 h-12 bg-gradient-to-r from-blue-500 to-purple-500 rounded-xl flex items-center justify-center">
                    <FiBriefcase className="text-white text-lg" />
                  </div>
                  <div className="text-xs text-gray-500 bg-gray-100 px-2 py-1 rounded-full">
                    Work
                  </div>
                </div>
                <div className="text-2xl font-bold text-gray-900 mb-1">
                  {experiences.length}
                </div>
                <p className="text-sm text-gray-600">Experience{experiences.length !== 1 ? 's' : ''}</p>
                <div className="mt-3 w-full bg-gray-200 rounded-full h-1">
                  <div 
                    className="h-1 bg-gradient-to-r from-blue-500 to-purple-500 rounded-full transition-all duration-1000"
                    style={{ width: `${Math.min(experiences.length * 20, 100)}%` }}
                  ></div>
                </div>
              </div>
            </div>

            <div className="group relative overflow-hidden bg-white/80 backdrop-blur-xl border border-white/40 rounded-2xl shadow-xl p-6 hover:shadow-2xl transition-all duration-500 hover:-translate-y-2">
              <div className="absolute inset-0 bg-gradient-to-br from-emerald-500/10 to-teal-500/10 opacity-0 group-hover:opacity-100 transition-opacity duration-500"></div>
              <div className="relative z-10">
                <div className="flex items-center justify-between mb-3">
                  <div className="w-12 h-12 bg-gradient-to-r from-emerald-500 to-teal-500 rounded-xl flex items-center justify-center">
                    <FiSettings className="text-white text-lg" />
                  </div>
                  <div className="text-xs text-gray-500 bg-gray-100 px-2 py-1 rounded-full">
                    Tech
                  </div>
                </div>
                <div className="text-2xl font-bold text-gray-900 mb-1">
                  {skills.length}
                </div>
                <p className="text-sm text-gray-600">Skill{skills.length !== 1 ? 's' : ''}</p>
                <div className="mt-3 w-full bg-gray-200 rounded-full h-1">
                  <div 
                    className="h-1 bg-gradient-to-r from-emerald-500 to-teal-500 rounded-full transition-all duration-1000"
                    style={{ width: `${Math.min(skills.length * 10, 100)}%` }}
                  ></div>
                </div>
              </div>
            </div>

            <div className="group relative overflow-hidden bg-white/80 backdrop-blur-xl border border-white/40 rounded-2xl shadow-xl p-6 hover:shadow-2xl transition-all duration-500 hover:-translate-y-2">
              <div className="absolute inset-0 bg-gradient-to-br from-orange-500/10 to-red-500/10 opacity-0 group-hover:opacity-100 transition-opacity duration-500"></div>
              <div className="relative z-10">
                <div className="flex items-center justify-between mb-3">
                  <div className="w-12 h-12 bg-gradient-to-r from-orange-500 to-red-500 rounded-xl flex items-center justify-center">
                    <FiBook className="text-white text-lg" />
                  </div>
                  <div className="text-xs text-gray-500 bg-gray-100 px-2 py-1 rounded-full">
                    Learn
                  </div>
                </div>
                <div className="text-2xl font-bold text-gray-900 mb-1">
                  {educations.length}
                </div>
                <p className="text-sm text-gray-600">Education{educations.length !== 1 ? 's' : ''}</p>
                <div className="mt-3 w-full bg-gray-200 rounded-full h-1">
                  <div 
                    className="h-1 bg-gradient-to-r from-orange-500 to-red-500 rounded-full transition-all duration-1000"
                    style={{ width: `${Math.min(educations.length * 25, 100)}%` }}
                  ></div>
                </div>
              </div>
            </div>

            <div className="group relative overflow-hidden bg-white/80 backdrop-blur-xl border border-white/40 rounded-2xl shadow-xl p-6 hover:shadow-2xl transition-all duration-500 hover:-translate-y-2">
              <div className="absolute inset-0 bg-gradient-to-br from-purple-500/10 to-pink-500/10 opacity-0 group-hover:opacity-100 transition-opacity duration-500"></div>
              <div className="relative z-10">
                <div className="flex items-center justify-between mb-3">
                  <div className="w-12 h-12 bg-gradient-to-r from-purple-500 to-pink-500 rounded-xl flex items-center justify-center">
                    <FiStar className="text-white text-lg" />
                  </div>
                  <div className="text-xs text-gray-500 bg-gray-100 px-2 py-1 rounded-full">
                    Score
                  </div>
                </div>
                <div className="text-2xl font-bold text-gray-900 mb-1">
                  {Math.round(((user?.profile?.full_name ? 1 : 0) + 
                              (user?.profile?.phone ? 1 : 0) + 
                              (user?.profile?.summary ? 1 : 0) + 
                              (user?.profile?.linkedin ? 1 : 0) + 
                              (experiences.length > 0 ? 1 : 0) + 
                              (skills.length > 0 ? 1 : 0) + 
                              (photoUrl ? 1 : 0)) / 7 * 100)}%
                </div>
                <p className="text-sm text-gray-600">Complete</p>
                <div className="mt-3 w-full bg-gray-200 rounded-full h-1">
                  <div 
                    className="h-1 bg-gradient-to-r from-purple-500 to-pink-500 rounded-full transition-all duration-1000"
                    style={{ 
                      width: `${Math.round(((user?.profile?.full_name ? 1 : 0) + 
                                          (user?.profile?.phone ? 1 : 0) + 
                                          (user?.profile?.summary ? 1 : 0) + 
                                          (user?.profile?.linkedin ? 1 : 0) + 
                                          (experiences.length > 0 ? 1 : 0) + 
                                          (skills.length > 0 ? 1 : 0) + 
                                          (photoUrl ? 1 : 0)) / 7 * 100)}%` 
                    }}
                  ></div>
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* Enhanced Status Messages */}
        {success && (
          <div className="max-w-4xl mx-auto mb-8">
            <div className="relative overflow-hidden rounded-2xl p-6 backdrop-blur-lg border bg-gradient-to-r from-emerald-50/80 to-green-50/80 border-emerald-200/50">
              <div className="relative z-10 flex items-center space-x-4">
                <div className="w-12 h-12 bg-emerald-500 rounded-xl flex items-center justify-center shadow-lg">
                  <FiCheck className="text-white text-xl" />
                </div>
                <div>
                  <h3 className="text-lg font-bold text-gray-900 mb-1">Success!</h3>
                  <p className="text-gray-700">{success}</p>
                </div>
              </div>
            </div>
          </div>
        )}

        {errors.general && (
          <div className="max-w-4xl mx-auto mb-8">
            <div className="relative overflow-hidden rounded-2xl p-6 backdrop-blur-lg border bg-gradient-to-r from-red-50/80 to-pink-50/80 border-red-200/50">
              <div className="relative z-10 flex items-center space-x-4">
                <div className="w-12 h-12 bg-red-500 rounded-xl flex items-center justify-center shadow-lg">
                  <FiAlertCircle className="text-white text-xl" />
                </div>
                <div>
                  <h3 className="text-lg font-bold text-gray-900 mb-1">Error</h3>
                  <p className="text-gray-700">{errors.general}</p>
                </div>
              </div>
            </div>
          </div>
        )}

        <div className="max-w-6xl mx-auto">
          <div className="grid lg:grid-cols-3 gap-8">
            {/* Enhanced Main Profile Form with Tabs */}
            <div className="lg:col-span-2 space-y-8">
              {/* Interactive Tab Navigation */}
              <div className="bg-white/80 backdrop-blur-xl border border-white/40 rounded-2xl shadow-xl overflow-hidden">
                <div className="flex flex-wrap bg-gradient-to-r from-gray-50 to-white border-b border-gray-200">
                  {[
                    { id: 'personal', label: 'Personal Info', icon: FiUser },
                    { id: 'experience', label: 'Experience', icon: FiBriefcase },
                    { id: 'skills', label: 'Skills', icon: FiSettings },
                    { id: 'education', label: 'Education', icon: FiBook }
                  ].map((tab) => (
                    <button
                      key={tab.id}
                      onClick={() => setActiveTab(tab.id)}
                      className={`flex-1 min-w-0 flex items-center justify-center space-x-2 px-6 py-4 text-sm font-medium transition-all duration-300 ${
                        activeTab === tab.id
                          ? 'bg-gradient-to-r from-blue-500 to-purple-500 text-white shadow-lg'
                          : 'text-gray-600 hover:text-gray-900 hover:bg-white'
                      }`}
                    >
                      <tab.icon className="text-sm" />
                      <span className="hidden sm:inline">{tab.label}</span>
                    </button>
                  ))}
                </div>
              </div>

              {/* Tab Content */}
              {activeTab === 'personal' && (
                <div>
                  {/* Basic Information Card */}
                  <div className="relative overflow-hidden bg-white/70 backdrop-blur-xl border border-white/30 rounded-3xl shadow-2xl p-8">
                <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-blue-500 to-purple-500"></div>
                <div className="flex items-center mb-8">
                  <div className="w-12 h-12 bg-gradient-to-r from-blue-500 to-indigo-500 rounded-xl flex items-center justify-center mr-4">
                    <FiUser className="text-white text-xl" />
                  </div>
                  <h2 className="text-2xl font-bold text-gray-900">Basic Information</h2>
                </div>
                
                <form onSubmit={handleSubmit} className="space-y-6">
                  <div className="grid md:grid-cols-2 gap-6">
                    <div className="space-y-2">
                      <label className="flex items-center text-sm font-semibold text-gray-700 mb-3">
                        <div className="w-5 h-5 bg-blue-100 rounded-lg flex items-center justify-center mr-3">
                          <FiUser className="text-blue-600 text-xs" />
                        </div>
                        Full Name
                      </label>
                      <input
                        type="text"
                        name="fullName"
                        value={formData.fullName}
                        onChange={handleChange}
                        className="w-full px-4 py-3 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white/50 backdrop-blur-sm transition-all duration-300"
                        required
                      />
                      {errors.full_name && <p className="text-red-500 text-sm mt-1 flex items-center space-x-1"><FiAlertCircle className="text-xs" /><span>{errors.full_name}</span></p>}
                    </div>

                    <div className="space-y-2">
                      <label className="flex items-center text-sm font-semibold text-gray-700 mb-3">
                        <div className="w-5 h-5 bg-purple-100 rounded-lg flex items-center justify-center mr-3">
                          <FiMail className="text-purple-600 text-xs" />
                        </div>
                        Email
                      </label>
                      <input
                        type="email"
                        name="email"
                        value={formData.email}
                        disabled
                        className="w-full px-4 py-3 border border-gray-200 rounded-xl bg-gray-100/70 cursor-not-allowed backdrop-blur-sm"
                      />
                    </div>

                    <div className="space-y-2">
                      <label className="flex items-center text-sm font-semibold text-gray-700 mb-3">
                        <div className="w-5 h-5 bg-emerald-100 rounded-lg flex items-center justify-center mr-3">
                          <FiPhone className="text-emerald-600 text-xs" />
                        </div>
                        Phone
                      </label>
                      <input
                        type="tel"
                        name="phone"
                        value={formData.phone}
                        onChange={handleChange}
                        className="w-full px-4 py-3 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white/50 backdrop-blur-sm transition-all duration-300"
                      />
                      {errors.phone && <p className="text-red-500 text-sm mt-1 flex items-center space-x-1"><FiAlertCircle className="text-xs" /><span>{errors.phone}</span></p>}
                    </div>

                    <div className="space-y-2">
                      <label className="flex items-center text-sm font-semibold text-gray-700 mb-3">
                        <div className="w-5 h-5 bg-blue-100 rounded-lg flex items-center justify-center mr-3">
                          <FiLinkedin className="text-blue-600 text-xs" />
                        </div>
                        LinkedIn URL
                      </label>
                      <input
                        type="url"
                        name="linkedin"
                        value={formData.linkedin}
                        onChange={handleChange}
                        className="w-full px-4 py-3 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white/50 backdrop-blur-sm transition-all duration-300"
                        placeholder="https://linkedin.com/in/yourprofile"
                      />
                      {errors.linkedin && <p className="text-red-500 text-sm mt-1 flex items-center space-x-1"><FiAlertCircle className="text-xs" /><span>{errors.linkedin}</span></p>}
                    </div>
                  </div>

                  {user?.role === 'student' && (
                    <>
                      <div className="grid md:grid-cols-2 gap-6">
                        <div className="space-y-2">
                          <label className="flex items-center text-sm font-semibold text-gray-700 mb-3">
                            <div className="w-5 h-5 bg-gray-100 rounded-lg flex items-center justify-center mr-3">
                              <FiGithub className="text-gray-600 text-xs" />
                            </div>
                            GitHub URL
                          </label>
                          <input
                            type="url"
                            name="github"
                            value={formData.github}
                            onChange={handleChange}
                            className="w-full px-4 py-3 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white/50 backdrop-blur-sm transition-all duration-300"
                            placeholder="https://github.com/yourusername"
                          />
                          {errors.github && <p className="text-red-500 text-sm mt-1 flex items-center space-x-1"><FiAlertCircle className="text-xs" /><span>{errors.github}</span></p>}
                        </div>

                        <div className="space-y-2">
                          <label className="flex items-center text-sm font-semibold text-gray-700 mb-3">
                            <div className="w-5 h-5 bg-indigo-100 rounded-lg flex items-center justify-center mr-3">
                              <FiBook className="text-indigo-600 text-xs" />
                            </div>
                            University
                          </label>
                          <input
                            type="text"
                            name="university"
                            value={formData.university}
                            onChange={handleChange}
                            className="w-full px-4 py-3 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white/50 backdrop-blur-sm transition-all duration-300"
                          />
                          {errors.university && <p className="text-red-500 text-sm mt-1 flex items-center space-x-1"><FiAlertCircle className="text-xs" /><span>{errors.university}</span></p>}
                        </div>
                      </div>

                      <div className="space-y-2">
                        <label className="flex items-center text-sm font-semibold text-gray-700 mb-3">
                          <div className="w-5 h-5 bg-yellow-100 rounded-lg flex items-center justify-center mr-3">
                            <FiBook className="text-yellow-600 text-xs" />
                          </div>
                          Graduation Year
                        </label>
                        <input
                          type="text"
                          name="graduationYear"
                          value={formData.graduationYear}
                          onChange={handleChange}
                          className="w-full px-4 py-3 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white/50 backdrop-blur-sm transition-all duration-300"
                          placeholder="2024"
                        />
                        {errors.graduation_year && <p className="text-red-500 text-sm mt-1 flex items-center space-x-1"><FiAlertCircle className="text-xs" /><span>{errors.graduation_year}</span></p>}
                      </div>
                    </>
                  )}

                  {user?.role === 'recruiter' && (
                    <div className="space-y-2">
                      <label className="flex items-center text-sm font-semibold text-gray-700 mb-3">
                        <div className="w-5 h-5 bg-orange-100 rounded-lg flex items-center justify-center mr-3">
                          <FiBriefcase className="text-orange-600 text-xs" />
                        </div>
                        Company Name
                      </label>
                      <input
                        type="text"
                        name="companyName"
                        value={formData.companyName}
                        onChange={handleChange}
                        className="w-full px-4 py-3 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white/50 backdrop-blur-sm transition-all duration-300"
                      />
                      {errors.company_name && <p className="text-red-500 text-sm mt-1 flex items-center space-x-1"><FiAlertCircle className="text-xs" /><span>{errors.company_name}</span></p>}
                    </div>
                  )}

                  <div className="space-y-2">
                    <label className="flex items-center text-sm font-semibold text-gray-700 mb-3">
                      <div className="w-5 h-5 bg-purple-100 rounded-lg flex items-center justify-center mr-3">
                        <FiTarget className="text-purple-600 text-xs" />
                      </div>
                      Professional Summary
                    </label>
                    <textarea
                      name="summary"
                      value={formData.summary}
                      onChange={handleChange}
                      rows={4}
                      className="w-full px-4 py-3 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white/50 backdrop-blur-sm transition-all duration-300 resize-none"
                      placeholder="Brief description of your professional background and goals..."
                    />
                    {errors.summary && <p className="text-red-500 text-sm mt-1 flex items-center space-x-1"><FiAlertCircle className="text-xs" /><span>{errors.summary}</span></p>}
                  </div>

                  <div className="space-y-2">
                    <label className="flex items-center text-sm font-semibold text-gray-700 mb-3">
                      <div className="w-5 h-5 bg-green-100 rounded-lg flex items-center justify-center mr-3">
                        <FiMapPin className="text-green-600 text-xs" />
                      </div>
                      Languages (comma-separated)
                    </label>
                    <input
                      type="text"
                      name="languages"
                      value={formData.languages}
                      onChange={handleChange}
                      className="w-full px-4 py-3 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white/50 backdrop-blur-sm transition-all duration-300"
                      placeholder="English, French, Spanish"
                    />
                    {errors.languages && <p className="text-red-500 text-sm mt-1 flex items-center space-x-1"><FiAlertCircle className="text-xs" /><span>{errors.languages}</span></p>}
                  </div>

                  <button
                    type="submit"
                    disabled={saving}
                    className="group relative w-full overflow-hidden bg-gradient-to-r from-blue-600 via-purple-600 to-indigo-600 text-white py-4 px-6 rounded-xl hover:shadow-2xl disabled:opacity-50 disabled:cursor-not-allowed transition-all duration-300 transform hover:scale-105"
                  >
                    <div className="absolute inset-0 bg-white/20 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                    <div className="relative flex items-center justify-center space-x-3">
                      {saving ? (
                        <>
                          <FiLoader className="animate-spin text-xl" />
                          <span className="font-semibold">Saving Profile...</span>
                        </>
                      ) : (
                        <>
                          <FiSave className="text-xl" />
                          <span className="font-semibold">Save Profile</span>
                        </>
                      )}
                    </div>
                  </button>
                </form>
              </div>
                </div>
              )}

              {/* Experience Tab */}
              {activeTab === 'experience' && user?.role === 'student' && (
                <div>
                  {/* Experience Section */}
                  <div className="relative overflow-hidden bg-white/70 backdrop-blur-xl border border-white/30 rounded-3xl shadow-2xl p-8">
                  <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-emerald-500 to-green-500"></div>
                  <div className="flex justify-between items-center mb-8">
                    <div className="flex items-center">
                      <div className="w-12 h-12 bg-gradient-to-r from-emerald-500 to-green-500 rounded-xl flex items-center justify-center mr-4">
                        <FiBriefcase className="text-white text-xl" />
                      </div>
                      <h2 className="text-2xl font-bold text-gray-900">Experience</h2>
                    </div>
                    <button
                      onClick={() => {
                        setEditingExperience(null);
                        setExperienceForm({ company: '', title: '', start_date: '', end_date: '', description: '' });
                        setShowExperienceModal(true);
                      }}
                      className="group relative overflow-hidden bg-gradient-to-r from-emerald-500 to-green-500 text-white px-6 py-3 rounded-xl hover:shadow-lg transition-all duration-300 transform hover:scale-105"
                    >
                      <div className="absolute inset-0 bg-white/20 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                      <div className="relative flex items-center space-x-2">
                        <FiPlus className="text-lg" />
                        <span className="font-semibold">Add Experience</span>
                      </div>
                    </button>
                  </div>

                  {experiencesLoading ? (
                    <div className="text-center py-12">
                      <div className="w-12 h-12 bg-emerald-100 rounded-xl flex items-center justify-center mx-auto mb-4">
                        <FiLoader className="animate-spin text-emerald-600 text-xl" />
                      </div>
                      <p className="text-gray-600">Loading experiences...</p>
                    </div>
                  ) : experiences.length === 0 ? (
                    <div className="text-center py-16">
                      <div className="w-20 h-20 bg-gray-100 rounded-2xl flex items-center justify-center mx-auto mb-6">
                        <FiBriefcase className="text-gray-400 text-3xl" />
                      </div>
                      <h3 className="text-xl font-semibold text-gray-900 mb-2">No Experience Added</h3>
                      <p className="text-gray-600 mb-6">Showcase your professional journey by adding your work experience.</p>
                    </div>
                  ) : (
                    <div className="space-y-6">
                      {experiences.map((exp) => (
                        <div key={exp.id} className="group relative overflow-hidden bg-gradient-to-r from-white to-gray-50 border border-gray-200 rounded-2xl p-6 hover:shadow-lg transition-all duration-300">
                          <div className="flex justify-between items-start">
                            <div className="flex-1">
                              <div className="flex items-start space-x-4 mb-4">
                                <div className="w-12 h-12 bg-gradient-to-r from-emerald-500 to-green-500 rounded-xl flex items-center justify-center flex-shrink-0">
                                  <FiBriefcase className="text-white text-lg" />
                                </div>
                                <div>
                                  <h3 className="text-lg font-bold text-gray-900">{exp.title}</h3>
                                  <p className="text-emerald-600 font-semibold">{exp.company}</p>
                                  <div className="flex items-center space-x-2 mt-2">
                                    <FiCalendar className="text-gray-400 text-sm" />
                                    <p className="text-sm text-gray-500">
                                      {new Date(exp.start_date).toLocaleDateString()} - 
                                      {exp.end_date ? new Date(exp.end_date).toLocaleDateString() : 'Present'}
                                    </p>
                                  </div>
                                </div>
                              </div>
                              {exp.description && (
                                <p className="text-gray-600 leading-relaxed bg-gray-50 p-4 rounded-xl">{exp.description}</p>
                              )}
                            </div>
                            <div className="flex space-x-3 ml-6 opacity-0 group-hover:opacity-100 transition-opacity duration-300">
                              <button
                                onClick={() => {
                                  setEditingExperience(exp);
                                  setExperienceForm(exp);
                                  setShowExperienceModal(true);
                                }}
                                className="w-10 h-10 bg-blue-100 hover:bg-blue-200 text-blue-600 rounded-lg flex items-center justify-center transition-colors duration-300"
                              >
                                <FiEdit3 className="text-sm" />
                              </button>
                              <button
                                onClick={() => exp.id && handleDeleteExperience(exp.id)}
                                className="w-10 h-10 bg-red-100 hover:bg-red-200 text-red-600 rounded-lg flex items-center justify-center transition-colors duration-300"
                              >
                                <FiTrash2 className="text-sm" />
                              </button>
                            </div>
                          </div>
                        </div>
                      ))}
                    </div>
                  )}
                </div>
                </div>
              )}

              {/* Skills Tab */}
              {activeTab === 'skills' && user?.role === 'student' && (
                <div>
                  {/* Skills Section */}
                  <div className="relative overflow-hidden bg-white/70 backdrop-blur-xl border border-white/30 rounded-3xl shadow-2xl p-8">
                  <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-purple-500 to-indigo-500"></div>
                  <div className="flex justify-between items-center mb-8">
                    <div className="flex items-center">
                      <div className="w-12 h-12 bg-gradient-to-r from-purple-500 to-indigo-500 rounded-xl flex items-center justify-center mr-4">
                        <FiSettings className="text-white text-xl" />
                      </div>
                      <h2 className="text-2xl font-bold text-gray-900">Skills</h2>
                    </div>
                    <button
                      onClick={() => {
                        setEditingSkill(null);
                        setSkillForm({ name: '', level: '' });
                        setShowSkillModal(true);
                      }}
                      className="group relative overflow-hidden bg-gradient-to-r from-purple-500 to-indigo-500 text-white px-6 py-3 rounded-xl hover:shadow-lg transition-all duration-300 transform hover:scale-105"
                    >
                      <div className="absolute inset-0 bg-white/20 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                      <div className="relative flex items-center space-x-2">
                        <FiPlus className="text-lg" />
                        <span className="font-semibold">Add Skill</span>
                      </div>
                    </button>
                  </div>

                  {skillsLoading ? (
                    <div className="text-center py-12">
                      <div className="w-12 h-12 bg-purple-100 rounded-xl flex items-center justify-center mx-auto mb-4">
                        <FiLoader className="animate-spin text-purple-600 text-xl" />
                      </div>
                      <p className="text-gray-600">Loading skills...</p>
                    </div>
                  ) : skills.length === 0 ? (
                    <div className="text-center py-16">
                      <div className="w-20 h-20 bg-gray-100 rounded-2xl flex items-center justify-center mx-auto mb-6">
                        <FiSettings className="text-gray-400 text-3xl" />
                      </div>
                      <h3 className="text-xl font-semibold text-gray-900 mb-2">No Skills Added</h3>
                      <p className="text-gray-600 mb-6">Highlight your technical and professional skills.</p>
                    </div>
                  ) : (
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                      {skills.map((skill) => (
                        <div key={skill.id} className="group relative overflow-hidden bg-gradient-to-r from-white to-purple-50 border border-purple-100 rounded-xl p-4 hover:shadow-lg transition-all duration-300">
                          <div className="flex justify-between items-center">
                            <div className="flex items-center space-x-3">
                              <div className="w-8 h-8 bg-gradient-to-r from-purple-500 to-indigo-500 rounded-lg flex items-center justify-center">
                                <FiStar className="text-white text-sm" />
                              </div>
                              <div>
                                <h4 className="font-semibold text-gray-900">{skill.name}</h4>
                                {skill.level && (
                                  <p className="text-sm text-purple-600 font-medium">{skill.level}</p>
                                )}
                              </div>
                            </div>
                            <div className="flex space-x-2 opacity-0 group-hover:opacity-100 transition-opacity duration-300">
                              <button
                                onClick={() => {
                                  setEditingSkill(skill);
                                  setSkillForm(skill);
                                  setShowSkillModal(true);
                                }}
                                className="w-8 h-8 bg-blue-100 hover:bg-blue-200 text-blue-600 rounded-lg flex items-center justify-center transition-colors duration-300"
                              >
                                <FiEdit3 className="text-xs" />
                              </button>
                              <button
                                onClick={() => skill.id && handleDeleteSkill(skill.id)}
                                className="w-8 h-8 bg-red-100 hover:bg-red-200 text-red-600 rounded-lg flex items-center justify-center transition-colors duration-300"
                              >
                                <FiTrash2 className="text-xs" />
                              </button>
                            </div>
                          </div>
                        </div>
                      ))}
                    </div>
                  )}
                </div>
                </div>
              )}

              {/* Education Tab */}
              {activeTab === 'education' && user?.role === 'student' && (
                <div>
                  {/* Education Section */}
                  <div className="relative overflow-hidden bg-white/70 backdrop-blur-xl border border-white/30 rounded-3xl shadow-2xl p-8">
                  <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-blue-500 to-indigo-500"></div>
                  <div className="flex justify-between items-center mb-8">
                    <div className="flex items-center">
                      <div className="w-12 h-12 bg-gradient-to-r from-blue-500 to-indigo-500 rounded-xl flex items-center justify-center mr-4">
                        <FiBook className="text-white text-xl" />
                      </div>
                      <h2 className="text-2xl font-bold text-gray-900">Education</h2>
                    </div>
                    <button
                      onClick={() => {
                        setEditingEducation(null);
                        setEducationForm({ school: '', degree: '', field: '', start_year: '', end_year: '', description: '' });
                        setShowEducationModal(true);
                      }}
                      className="group relative overflow-hidden bg-gradient-to-r from-blue-500 to-indigo-500 text-white px-6 py-3 rounded-xl hover:shadow-lg transition-all duration-300 transform hover:scale-105"
                    >
                      <div className="absolute inset-0 bg-white/20 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                      <div className="relative flex items-center space-x-2">
                        <FiPlus className="text-lg" />
                        <span className="font-semibold">Add Education</span>
                      </div>
                    </button>
                  </div>

                  {educationsLoading ? (
                    <div className="text-center py-12">
                      <div className="w-12 h-12 bg-blue-100 rounded-xl flex items-center justify-center mx-auto mb-4">
                        <FiLoader className="animate-spin text-blue-600 text-xl" />
                      </div>
                      <p className="text-gray-600">Loading education...</p>
                    </div>
                  ) : educations.length === 0 ? (
                    <div className="text-center py-16">
                      <div className="w-20 h-20 bg-gray-100 rounded-2xl flex items-center justify-center mx-auto mb-6">
                        <FiBook className="text-gray-400 text-3xl" />
                      </div>
                      <h3 className="text-xl font-semibold text-gray-900 mb-2">No Education Added</h3>
                      <p className="text-gray-600 mb-6">Add your educational background to complete your profile.</p>
                    </div>
                  ) : (
                    <div className="space-y-6">
                      {educations.map((education) => (
                        <div key={education.id} className="group relative overflow-hidden bg-gradient-to-r from-white to-blue-50 border border-blue-100 rounded-2xl p-6 hover:shadow-lg transition-all duration-300">
                          <div className="flex justify-between items-start">
                            <div className="flex-1">
                              <div className="flex items-start space-x-4 mb-4">
                                <div className="w-12 h-12 bg-gradient-to-r from-blue-500 to-indigo-500 rounded-xl flex items-center justify-center flex-shrink-0">
                                  <FiBook className="text-white text-lg" />
                                </div>
                                <div>
                                  <h3 className="text-lg font-bold text-gray-900">{education.school}</h3>
                                  <p className="text-blue-600 font-semibold">
                                    {education.degree} {education.field && `in ${education.field}`}
                                  </p>
                                  <div className="flex items-center space-x-2 mt-2">
                                    <FiCalendar className="text-gray-400 text-sm" />
                                    <p className="text-sm text-gray-500">
                                      {education.start_year} - {education.end_year || 'Present'}
                                    </p>
                                  </div>
                                </div>
                              </div>
                              {education.description && (
                                <p className="text-gray-600 leading-relaxed bg-gray-50 p-4 rounded-xl">{education.description}</p>
                              )}
                            </div>
                            <div className="flex space-x-3 ml-6 opacity-0 group-hover:opacity-100 transition-opacity duration-300">
                              <button
                                onClick={() => {
                                  setEditingEducation(education);
                                  setEducationForm(education);
                                  setShowEducationModal(true);
                                }}
                                className="w-10 h-10 bg-blue-100 hover:bg-blue-200 text-blue-600 rounded-lg flex items-center justify-center transition-colors duration-300"
                              >
                                <FiEdit3 className="text-sm" />
                              </button>
                              <button
                                onClick={() => education.id && handleDeleteEducation(education.id)}
                                className="w-10 h-10 bg-red-100 hover:bg-red-200 text-red-600 rounded-lg flex items-center justify-center transition-colors duration-300"
                              >
                                <FiTrash2 className="text-sm" />
                              </button>
                            </div>
                          </div>
                        </div>
                      ))}
                    </div>
                  )}
                </div>
                </div>
              )}
            </div>

          {/* Sidebar */}
          <div className="space-y-6">
            {/* Ultra-Enhanced Profile Photo */}
            <div className="relative overflow-hidden bg-white/80 backdrop-blur-xl border border-white/40 rounded-2xl shadow-xl p-6">
              <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-pink-500 via-purple-500 to-indigo-500"></div>
              <div className="flex items-center justify-between mb-6">
                <div className="flex items-center">
                  <div className="w-8 h-8 bg-gradient-to-r from-pink-500 to-purple-500 rounded-lg flex items-center justify-center mr-3">
                    <FiUser className="text-white text-sm" />
                  </div>
                  <h3 className="text-lg font-semibold text-gray-900">Profile Photo</h3>
                </div>
                {photoUrl && (
                  <div className="flex items-center space-x-2 text-xs text-gray-500">
                    <FiCheck className="text-emerald-500" />
                    <span>Verified</span>
                  </div>
                )}
              </div>
              
              <div className="text-center">
                {/* Enhanced Photo Display with Stats */}
                <div className="relative inline-block group mb-6">
                  <div className="relative">
                    {photoUrl ? (
                      <div className="relative">
                        {/* Main Photo */}
                        <img
                          src={photoUrl}
                          alt="Profile"
                          className="w-36 h-36 rounded-full object-cover border-4 border-white shadow-2xl group-hover:shadow-3xl transition-all duration-500 transform group-hover:scale-105"
                          onError={(e) => {
                            console.error('Image failed to load:', photoUrl);
                            console.error('Error event:', e);
                          }}
                          onLoad={() => {
                            console.log('Image loaded successfully:', photoUrl);
                          }}
                        />
                        
                        {/* Hover Overlay with Actions */}
                        <div className="absolute inset-0 bg-gradient-to-t from-black/60 via-transparent to-transparent rounded-full opacity-0 group-hover:opacity-100 transition-all duration-300">
                          <div className="absolute bottom-2 left-1/2 transform -translate-x-1/2 flex space-x-2">
                            <button className="w-8 h-8 bg-white/90 rounded-full flex items-center justify-center text-gray-700 hover:bg-white transition-colors">
                              <FiEdit3 className="text-xs" />
                            </button>
                            <button className="w-8 h-8 bg-white/90 rounded-full flex items-center justify-center text-red-600 hover:bg-white transition-colors">
                              <FiTrash2 className="text-xs" />
                            </button>
                          </div>
                        </div>
                        
                        {/* Status Badge */}
                        <div className="absolute -top-1 -right-1 w-6 h-6 bg-emerald-500 rounded-full border-2 border-white flex items-center justify-center">
                          <FiCheck className="text-white text-xs" />
                        </div>
                      </div>
                    ) : (
                      <div className="w-36 h-36 rounded-full bg-gradient-to-br from-gray-100 via-gray-200 to-gray-300 flex items-center justify-center text-gray-500 border-4 border-white shadow-2xl group-hover:shadow-3xl transition-all duration-500 transform group-hover:scale-105 relative overflow-hidden">
                        {/* Animated Background */}
                        <div className="absolute inset-0 bg-gradient-to-br from-blue-500/10 via-purple-500/10 to-pink-500/10 opacity-0 group-hover:opacity-100 transition-opacity duration-500"></div>
                        
                        {/* Icon */}
                        <div className="relative z-10 flex flex-col items-center">
                          <FiUser className="text-4xl mb-2" />
                          <span className="text-xs font-medium">Add Photo</span>
                        </div>
                        
                        {/* Floating Elements */}
                        <div className="absolute inset-0 opacity-0 group-hover:opacity-100 transition-opacity duration-500">
                          <div className="absolute top-4 left-4 w-2 h-2 bg-blue-400 rounded-full animate-pulse"></div>
                          <div className="absolute top-8 right-6 w-1.5 h-1.5 bg-purple-400 rounded-full animate-pulse delay-300"></div>
                          <div className="absolute bottom-6 left-8 w-1 h-1 bg-pink-400 rounded-full animate-pulse delay-500"></div>
                        </div>
                      </div>
                    )}
                    
                    {/* Animated Ring */}
                    <div className="absolute -inset-2 bg-gradient-to-r from-pink-500 via-purple-500 to-indigo-500 rounded-full blur-sm opacity-30 group-hover:opacity-60 group-hover:blur-md transition-all duration-500 animate-pulse"></div>
                  </div>
                </div>

                {/* Photo Statistics */}
                {photoUrl && (
                  <div className="grid grid-cols-3 gap-3 mb-6">
                    <div className="bg-gradient-to-br from-emerald-50 to-green-50 border border-emerald-200 rounded-lg p-3">
                      <div className="text-emerald-600 text-xs font-semibold">Visibility</div>
                      <div className="text-emerald-800 text-lg font-bold">+340%</div>
                    </div>
                    <div className="bg-gradient-to-br from-blue-50 to-indigo-50 border border-blue-200 rounded-lg p-3">
                      <div className="text-blue-600 text-xs font-semibold">Quality</div>
                      <div className="text-blue-800 text-lg font-bold">9.2/10</div>
                    </div>
                    <div className="bg-gradient-to-br from-purple-50 to-pink-50 border border-purple-200 rounded-lg p-3">
                      <div className="text-purple-600 text-xs font-semibold">Impact</div>
                      <div className="text-purple-800 text-lg font-bold">High</div>
                    </div>
                  </div>
                )}

                {/* Enhanced Upload Section */}
                <div className="space-y-4">
                  <label className="block">
                    <span className="sr-only">Choose photo</span>
                    <input
                      type="file"
                      accept="image/*"
                      onChange={handlePhotoUpload}
                      className="hidden"
                      disabled={photoUploading}
                    />
                    <button
                      type="button"
                      onClick={() => {
                        const fileInput = document.querySelector('input[type="file"]') as HTMLInputElement;
                        fileInput?.click();
                      }}
                      disabled={photoUploading}
                      className="w-full bg-gradient-to-r from-pink-500 via-purple-500 to-indigo-500 text-white py-3 px-6 rounded-xl hover:from-pink-600 hover:via-purple-600 hover:to-indigo-600 disabled:opacity-50 transition-all duration-300 shadow-lg hover:shadow-xl transform hover:-translate-y-0.5 flex items-center justify-center space-x-2 group"
                    >
                      {photoUploading ? (
                        <>
                          <FiLoader className="animate-spin" />
                          <span>Processing...</span>
                        </>
                      ) : (
                        <>
                          <FiPlus className="group-hover:rotate-90 transition-transform duration-300" />
                          <span>{photoUrl ? 'Change Photo' : 'Upload Photo'}</span>
                        </>
                      )}
                    </button>
                  </label>
                  
                  {/* Drag & Drop Zone */}
                  <div className="border-2 border-dashed border-gray-300 rounded-lg p-4 text-center hover:border-purple-400 hover:bg-purple-50/50 transition-all duration-300 cursor-pointer">
                    <FiUpload className="mx-auto text-gray-400 text-xl mb-2" />
                    <p className="text-xs text-gray-600">
                      Drag & drop your photo here, or{' '}
                      <span className="text-purple-600 font-medium">browse files</span>
                    </p>
                  </div>
                  
                  {errors.photo && (
                    <div className="flex items-center space-x-2 text-red-500 text-sm bg-red-50 border border-red-200 rounded-lg p-3">
                      <FiAlertCircle className="text-sm flex-shrink-0" />
                      <span>{errors.photo}</span>
                    </div>
                  )}
                  
                  {/* Enhanced Tips */}
                  <div className="space-y-3">
                    <div className="bg-gradient-to-r from-blue-50 to-indigo-50 border border-blue-200 rounded-lg p-4">
                      <div className="flex items-start space-x-3">
                        <FiInfo className="text-blue-600 text-sm mt-0.5 flex-shrink-0" />
                        <div>
                          <h4 className="text-blue-900 font-semibold text-sm mb-1">Photo Guidelines</h4>
                          <ul className="text-blue-700 text-xs space-y-1">
                            <li>• Use a clear, professional headshot</li>
                            <li>• Face should take up 60-70% of the frame</li>
                            <li>• Avoid filters, sunglasses, or group photos</li>
                            <li>• Recommended: 400x400px minimum, Max: 5MB</li>
                          </ul>
                        </div>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            </div>

            {/* Profile Completion Guide */}
            <div className="relative overflow-hidden bg-white/80 backdrop-blur-xl border border-white/40 rounded-2xl shadow-xl p-6">
              <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-emerald-500 to-teal-500"></div>
              <div className="flex items-center mb-6">
                <div className="w-8 h-8 bg-gradient-to-r from-emerald-500 to-teal-500 rounded-lg flex items-center justify-center mr-3">
                  <FiTarget className="text-white text-sm" />
                </div>
                <h3 className="text-lg font-semibold text-gray-900">Quick Tips</h3>
              </div>
              
              <div className="space-y-4">
                {tipsLoading ? (
                  <div className="flex items-center justify-center p-8">
                    <FiLoader className="animate-spin text-blue-500 text-2xl" />
                  </div>
                ) : aiTips.length === 0 ? (
                  <div className="flex items-start space-x-3 p-3 bg-gray-50 rounded-lg border border-gray-200">
                    <div className="w-6 h-6 bg-gray-400 rounded-full flex items-center justify-center flex-shrink-0 mt-0.5">
                      <FiInfo className="text-white text-xs" />
                    </div>
                    <div>
                      <p className="text-sm font-medium text-gray-900">No tips available</p>
                      <p className="text-xs text-gray-600">Complete your profile to get personalized suggestions</p>
                    </div>
                  </div>
                ) : (
                  aiTips.slice(0, 5).map((tip, index) => {
                    const categoryColors: { [key: string]: { bg: string; border: string; text: string; iconBg: string } } = {
                      profile: { bg: 'bg-emerald-50', border: 'border-emerald-200', text: 'text-emerald-900', iconBg: 'bg-emerald-500' },
                      skills: { bg: 'bg-orange-50', border: 'border-orange-200', text: 'text-orange-900', iconBg: 'bg-orange-500' },
                      experience: { bg: 'bg-purple-50', border: 'border-purple-200', text: 'text-purple-900', iconBg: 'bg-purple-500' },
                      branding: { bg: 'bg-blue-50', border: 'border-blue-200', text: 'text-blue-900', iconBg: 'bg-blue-500' },
                      readiness: { bg: 'bg-pink-50', border: 'border-pink-200', text: 'text-pink-900', iconBg: 'bg-pink-500' },
                    };
                    
                    const colors = categoryColors[tip.category || 'profile'] || categoryColors.profile;
                    
                    return (
                      <div key={index} className={`flex items-start space-x-3 p-3 ${colors.bg} rounded-lg border ${colors.border}`}>
                        <div className={`w-6 h-6 ${colors.iconBg} rounded-full flex items-center justify-center flex-shrink-0 mt-0.5`}>
                          <FiCheck className="text-white text-xs" />
                        </div>
                        <div className="flex-1">
                          <div className="flex items-start justify-between">
                            <p className={`text-sm font-medium ${colors.text}`}>{tip.title}</p>
                            {tip.priority && (
                              <span className={`text-[10px] px-2 py-0.5 rounded-full font-semibold ${
                                tip.priority === 'high' ? 'bg-red-100 text-red-700' :
                                tip.priority === 'low' ? 'bg-gray-100 text-gray-600' :
                                'bg-yellow-100 text-yellow-700'
                              }`}>
                                {tip.priority.toUpperCase()}
                              </span>
                            )}
                          </div>
                          <p className={`text-xs ${colors.text} opacity-80 mt-1`}>{tip.description}</p>
                          {tip.action && (
                            <p className={`text-xs ${colors.text} font-medium mt-1.5 italic`}>→ {tip.action}</p>
                          )}
                        </div>
                      </div>
                    );
                  })
                )}
              </div>

              {/* Progress indicator */}
              <div className="mt-6 p-4 bg-gradient-to-r from-blue-50 to-purple-50 rounded-lg border border-blue-200">
                <div className="flex items-center justify-between mb-2">
                  <span className="text-sm font-medium text-gray-700">Profile Strength</span>
                  <span className="text-sm font-bold text-blue-600">
                    {Math.round(((user?.profile?.full_name ? 1 : 0) + 
                                (user?.profile?.phone ? 1 : 0) + 
                                (user?.profile?.summary ? 1 : 0) + 
                                (user?.profile?.linkedin ? 1 : 0) + 
                                (experiences.length > 0 ? 1 : 0) + 
                                (skills.length > 0 ? 1 : 0) + 
                                (photoUrl ? 1 : 0)) / 7 * 100)}%
                  </span>
                </div>
                <div className="w-full bg-gray-200 rounded-full h-2">
                  <div 
                    className="h-2 bg-gradient-to-r from-blue-500 to-purple-500 rounded-full transition-all duration-1000"
                    style={{ 
                      width: `${Math.round(((user?.profile?.full_name ? 1 : 0) + 
                                          (user?.profile?.phone ? 1 : 0) + 
                                          (user?.profile?.summary ? 1 : 0) + 
                                          (user?.profile?.linkedin ? 1 : 0) + 
                                          (experiences.length > 0 ? 1 : 0) + 
                                          (skills.length > 0 ? 1 : 0) + 
                                          (photoUrl ? 1 : 0)) / 7 * 100)}%` 
                    }}
                  ></div>
                </div>
                <p className="text-xs text-gray-600 mt-2 text-center">
                  {Math.round(((user?.profile?.full_name ? 1 : 0) + 
                              (user?.profile?.phone ? 1 : 0) + 
                              (user?.profile?.summary ? 1 : 0) + 
                              (user?.profile?.linkedin ? 1 : 0) + 
                              (experiences.length > 0 ? 1 : 0) + 
                              (skills.length > 0 ? 1 : 0) + 
                              (photoUrl ? 1 : 0)) / 7 * 100) >= 80 
                    ? 'Excellent! Your profile looks professional.' 
                    : Math.round(((user?.profile?.full_name ? 1 : 0) + 
                                 (user?.profile?.phone ? 1 : 0) + 
                                 (user?.profile?.summary ? 1 : 0) + 
                                 (user?.profile?.linkedin ? 1 : 0) + 
                                 (experiences.length > 0 ? 1 : 0) + 
                                 (skills.length > 0 ? 1 : 0) + 
                                 (photoUrl ? 1 : 0)) / 7 * 100) >= 50 
                      ? 'Good progress! Keep adding details.' 
                      : 'Getting started! Add more information to boost visibility.'
                  }
                </p>
              </div>
              </div>
            </div>
          </div>
        </div>

        {/* Experience Modal */}
        {showExperienceModal && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white rounded-lg p-6 w-full max-w-2xl mx-4">
              <div className="flex justify-between items-center mb-4">
                <h3 className="text-lg font-semibold">
                  {editingExperience ? 'Edit Experience' : 'Add Experience'}
                </h3>
                <button
                  onClick={() => setShowExperienceModal(false)}
                  className="text-gray-500 hover:text-gray-700"
                >
                  <FiX />
                </button>
              </div>
              
              <div className="space-y-4">
                <div className="grid md:grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Company *
                    </label>
                    <input
                      type="text"
                      value={experienceForm.company}
                      onChange={(e) => setExperienceForm(prev => ({ ...prev, company: e.target.value }))}
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                      required
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Position *
                    </label>
                    <input
                      type="text"
                      value={experienceForm.title}
                      onChange={(e) => setExperienceForm(prev => ({ ...prev, title: e.target.value }))}
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                      required
                    />
                  </div>
                </div>
                
                <div className="grid md:grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Start Date *
                    </label>
                    <input
                      type="date"
                      value={experienceForm.start_date}
                      onChange={(e) => setExperienceForm(prev => ({ ...prev, start_date: e.target.value }))}
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                      required
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      End Date (Leave empty if current)
                    </label>
                    <input
                      type="date"
                      value={experienceForm.end_date || ''}
                      onChange={(e) => setExperienceForm(prev => ({ ...prev, end_date: e.target.value }))}
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                    />
                  </div>
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Description
                  </label>
                  <textarea
                    value={experienceForm.description || ''}
                    onChange={(e) => setExperienceForm(prev => ({ ...prev, description: e.target.value }))}
                    rows={3}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                    placeholder="Describe your responsibilities and achievements..."
                  />
                </div>
              </div>
              
              <div className="flex justify-end space-x-3 mt-6">
                <button
                  onClick={() => setShowExperienceModal(false)}
                  className="px-4 py-2 text-gray-600 border border-gray-300 rounded-lg hover:bg-gray-50"
                >
                  Cancel
                </button>
                <button
                  onClick={handleSaveExperience}
                  className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
                >
                  Save Experience
                </button>
              </div>
            </div>
          </div>
        )}

        {/* Skill Modal */}
        {showSkillModal && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white rounded-lg p-6 w-full max-w-md mx-4">
              <div className="flex justify-between items-center mb-4">
                <h3 className="text-lg font-semibold">
                  {editingSkill ? 'Edit Skill' : 'Add Skill'}
                </h3>
                <button
                  onClick={() => setShowSkillModal(false)}
                  className="text-gray-500 hover:text-gray-700"
                >
                  <FiX />
                </button>
              </div>
              
              <div className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Skill Name *
                  </label>
                  <input
                    type="text"
                    value={skillForm.name}
                    onChange={(e) => setSkillForm(prev => ({ ...prev, name: e.target.value }))}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                    placeholder="e.g., JavaScript, React, Python"
                    required
                  />
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Proficiency Level
                  </label>
                  <select
                    value={skillForm.level || ''}
                    onChange={(e) => setSkillForm(prev => ({ ...prev, level: e.target.value }))}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                  >
                    <option value="">Select level</option>
                    <option value="Beginner">Beginner</option>
                    <option value="Intermediate">Intermediate</option>
                    <option value="Advanced">Advanced</option>
                    <option value="Expert">Expert</option>
                  </select>
                </div>
              </div>
              
              <div className="flex justify-end space-x-3 mt-6">
                <button
                  onClick={() => setShowSkillModal(false)}
                  className="px-4 py-2 text-gray-600 border border-gray-300 rounded-lg hover:bg-gray-50"
                >
                  Cancel
                </button>
                <button
                  onClick={handleSaveSkill}
                  className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
                >
                  Save Skill
                </button>
              </div>
            </div>
          </div>
        )}

        {/* Education Modal */}
        {showEducationModal && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white rounded-lg p-6 w-full max-w-md mx-4">
              <div className="flex items-center justify-between mb-4">
                <h3 className="text-lg font-semibold">
                  {editingEducation ? 'Edit Education' : 'Add Education'}
                </h3>
                <button
                  onClick={() => setShowEducationModal(false)}
                  className="text-gray-400 hover:text-gray-600"
                >
                  <FiX />
                </button>
              </div>
              
              <div className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    School/University *
                  </label>
                  <input
                    type="text"
                    value={educationForm.school}
                    onChange={(e) => setEducationForm(prev => ({ ...prev, school: e.target.value }))}
                    placeholder="Enter school or university name"
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-purple-500 outline-none"
                    required
                  />
                </div>
                
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Degree
                    </label>
                    <input
                      type="text"
                      value={educationForm.degree || ''}
                      onChange={(e) => setEducationForm(prev => ({ ...prev, degree: e.target.value }))}
                      placeholder="e.g., Bachelor's"
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-purple-500 outline-none"
                    />
                  </div>
                  
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Field of Study
                    </label>
                    <input
                      type="text"
                      value={educationForm.field || ''}
                      onChange={(e) => setEducationForm(prev => ({ ...prev, field: e.target.value }))}
                      placeholder="e.g., Computer Science"
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-purple-500 outline-none"
                    />
                  </div>
                </div>
                
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Start Year
                    </label>
                    <input
                      type="text"
                      value={educationForm.start_year || ''}
                      onChange={(e) => setEducationForm(prev => ({ ...prev, start_year: e.target.value }))}
                      placeholder="e.g., 2020"
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-purple-500 outline-none"
                    />
                  </div>
                  
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      End Year
                    </label>
                    <input
                      type="text"
                      value={educationForm.end_year || ''}
                      onChange={(e) => setEducationForm(prev => ({ ...prev, end_year: e.target.value }))}
                      placeholder="e.g., 2024"
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-purple-500 outline-none"
                    />
                  </div>
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Description (Optional)
                  </label>
                  <textarea
                    value={educationForm.description || ''}
                    onChange={(e) => setEducationForm(prev => ({ ...prev, description: e.target.value }))}
                    placeholder="Additional details about your education"
                    rows={3}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-purple-500 outline-none resize-none"
                  />
                </div>
              </div>
              
              <div className="flex justify-end space-x-3 mt-6">
                <button
                  onClick={() => setShowEducationModal(false)}
                  className="px-4 py-2 text-gray-600 border border-gray-300 rounded-lg hover:bg-gray-50"
                >
                  Cancel
                </button>
                <button
                  onClick={handleSaveEducation}
                  className="px-4 py-2 bg-purple-600 text-white rounded-lg hover:bg-purple-700"
                >
                  Save Education
                </button>
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

export default Profile;