import React, { useState, useEffect, useCallback } from 'react';
import { useNavigate } from 'react-router-dom';
import apiClient from '../apiClient';
import { 
  FiUser, 
  FiLoader, 
  FiArrowLeft, 
  FiGlobe,
  FiBriefcase,
  FiMail,
  FiUsers,
  FiMapPin,
  FiTarget,
  FiCheck,
  FiFileText
} from 'react-icons/fi';

interface ProfileData {
  full_name?: string;
  email?: string;
  position?: string;
  department?: string;
  company?: {
    name?: string;
    description?: string;
    industry?: string;
    size?: string;
    website?: string;
    logo_url?: string;
  };
}

const ProfileRecruiter: React.FC = () => {
  const [loading, setLoading] = useState(true);
  const [profileData, setProfileData] = useState<ProfileData | null>(null);
  const navigate = useNavigate();

  const fetchUserData = useCallback(async () => {
    try {
      const response = await apiClient.get('/api/user');
      const userData = response.data.data?.user || response.data.user;
      
      setProfileData({
        full_name: userData.profile?.full_name,
        email: userData.email,
        position: userData.profile?.position,
        department: userData.profile?.department,
        company: userData.profile?.company || {
          name: userData.profile?.company_name
        }
      });
    } catch (error) {
      console.error('Error fetching user data:', error);
      navigate('/login');
    } finally {
      setLoading(false);
    }
  }, [navigate]);

  useEffect(() => {
    fetchUserData();
  }, [fetchUserData]);

  if (loading) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-100 flex items-center justify-center">
        <div className="text-center">
          <div className="relative">
            <div className="w-16 h-16 bg-gradient-to-r from-blue-600 via-purple-600 to-indigo-600 rounded-2xl flex items-center justify-center shadow-xl mx-auto mb-6">
              <FiLoader className="animate-spin text-white text-2xl" />
            </div>
            <div className="absolute inset-0 bg-gradient-to-r from-blue-600 via-purple-600 to-indigo-600 rounded-2xl blur-lg opacity-30 animate-pulse"></div>
          </div>
          <h3 className="text-xl font-bold text-gray-900 mb-2">Loading Your Profile</h3>
          <p className="text-gray-600">Preparing your professional dashboard...</p>
        </div>
      </div>
    );
  }

  const completionItems = [
    { label: 'Full Name', completed: !!profileData?.full_name, icon: FiUser },
    { label: 'Position', completed: !!profileData?.position, icon: FiBriefcase },
    { label: 'Department', completed: !!profileData?.department, icon: FiUsers },
    { label: 'Company', completed: !!profileData?.company?.name, icon: FiFileText },
    { label: 'Industry', completed: !!profileData?.company?.industry, icon: FiMapPin },
    { label: 'Website', completed: !!profileData?.company?.website, icon: FiGlobe },
  ];

  const completionPercentage = Math.round((completionItems.filter(item => item.completed).length / completionItems.length) * 100);

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-100 py-12">
      <div className="max-w-7xl mx-auto px-4">
        {/* Ultra Modern Header */}
        <div className="text-center mb-12">
          <div className="flex items-center justify-center mb-6">
            <div className="relative">
              <div className="w-16 h-16 bg-gradient-to-r from-blue-600 via-purple-600 to-indigo-600 rounded-2xl flex items-center justify-center shadow-xl">
                <FiUser className="text-white text-2xl" />
              </div>
              <div className="absolute -inset-1 bg-gradient-to-r from-blue-600 via-purple-600 to-indigo-600 rounded-2xl blur-lg opacity-30 animate-pulse"></div>
            </div>
          </div>
          <div className="flex items-center justify-center space-x-4 mb-6">
            <button
              onClick={() => navigate(-1)}
              className="group flex items-center space-x-2 text-gray-600 hover:text-blue-600 transition-colors duration-300"
            >
              <div className="w-8 h-8 bg-gray-100 group-hover:bg-blue-50 rounded-lg flex items-center justify-center transition-colors duration-300">
                <FiArrowLeft className="text-sm" />
              </div>
              <span className="font-medium">Back</span>
            </button>
            <div className="w-px h-6 bg-gray-300"></div>
            <h1 className="text-5xl font-bold">
              <span className="bg-gradient-to-r from-blue-600 via-purple-600 to-indigo-600 bg-clip-text text-transparent">
                Recruiter Profile
              </span>
            </h1>
          </div>
          <p className="text-xl text-gray-600 max-w-3xl mx-auto leading-relaxed">
            View your professional information and company details.
            <br />
            <span className="text-gray-500">Contact support to update your information if needed.</span>
          </p>
        </div>

        {/* Advanced Profile Completion Tracker */}
        <div className="max-w-4xl mx-auto mb-12">
          <div className="relative overflow-hidden bg-white/80 backdrop-blur-xl border border-white/40 rounded-3xl shadow-2xl p-8">
            <div className="absolute inset-0 bg-gradient-to-br from-blue-500/5 via-purple-500/5 to-indigo-500/5"></div>
            <div className="relative z-10">
              <div className="flex items-center justify-between mb-6">
                <div className="flex items-center space-x-4">
                  <div className="w-14 h-14 bg-gradient-to-r from-blue-600 to-purple-600 rounded-xl flex items-center justify-center shadow-lg">
                    <FiTarget className="text-white text-xl" />
                  </div>
                  <div>
                    <h3 className="text-2xl font-bold text-gray-900">Profile Completion</h3>
                    <p className="text-gray-600">Your profile completeness status</p>
                  </div>
                </div>
                <div className="text-right">
                  <div className="text-3xl font-bold bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent">
                    {completionPercentage}%
                  </div>
                  <p className="text-sm text-gray-500">Complete</p>
                </div>
              </div>
              
              {/* Enhanced Progress Bar */}
              <div className="mb-6">
                <div className="w-full bg-gray-200 rounded-full h-3 overflow-hidden">
                  <div 
                    className="h-full bg-gradient-to-r from-blue-500 via-purple-500 to-indigo-500 rounded-full transition-all duration-1000 ease-out relative"
                    style={{ width: `${completionPercentage}%` }}
                  >
                    <div className="absolute inset-0 bg-white/30 animate-pulse"></div>
                  </div>
                </div>
              </div>

              {/* Interactive Completion Checklist */}
              <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-6 gap-4">
                {completionItems.map((item, index) => (
                  <div 
                    key={index}
                    className={`p-4 rounded-xl border-2 transition-all duration-300 ${
                      item.completed 
                        ? 'bg-emerald-50 border-emerald-200 hover:bg-emerald-100' 
                        : 'bg-gray-50 border-gray-200'
                    }`}
                  >
                    <div className="flex flex-col items-center space-y-2">
                      <div className={`w-8 h-8 rounded-lg flex items-center justify-center ${
                        item.completed ? 'bg-emerald-500' : 'bg-gray-300'
                      }`}>
                        {item.completed ? (
                          <FiCheck className="text-white text-sm" />
                        ) : (
                          <item.icon className="text-gray-600 text-sm" />
                        )}
                      </div>
                      <span className={`text-xs font-medium text-center ${
                        item.completed ? 'text-emerald-700' : 'text-gray-600'
                      }`}>
                        {item.label}
                      </span>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </div>
        </div>

        {/* Profile Content */}
        <div className="max-w-4xl mx-auto">
          <div className="bg-white/70 backdrop-blur-xl border border-white/30 rounded-3xl shadow-2xl overflow-hidden">
            <div className="w-full h-1 bg-gradient-to-r from-blue-500 to-purple-500"></div>
            
            <div className="p-8">
              {/* Personal Information Section */}
              <div className="mb-10">
                <div className="flex items-center mb-8">
                  <div className="w-12 h-12 bg-gradient-to-r from-blue-500 to-indigo-500 rounded-xl flex items-center justify-center mr-4">
                    <FiUser className="text-white text-xl" />
                  </div>
                  <h2 className="text-2xl font-bold text-gray-900">Personal Information</h2>
                </div>
                
                <div className="grid md:grid-cols-2 gap-6">
                  <div className="space-y-2">
                    <label className="flex items-center text-sm font-semibold text-gray-700 mb-3">
                      <div className="w-5 h-5 bg-blue-100 rounded-lg flex items-center justify-center mr-3">
                        <FiUser className="text-blue-600 text-xs" />
                      </div>
                      Full Name
                    </label>
                    <div className="w-full px-4 py-3 border border-gray-200 rounded-xl bg-gray-50/70 backdrop-blur-sm">
                      <p className="text-gray-900 font-medium">{profileData?.full_name || 'Not provided'}</p>
                    </div>
                  </div>

                  <div className="space-y-2">
                    <label className="flex items-center text-sm font-semibold text-gray-700 mb-3">
                      <div className="w-5 h-5 bg-purple-100 rounded-lg flex items-center justify-center mr-3">
                        <FiMail className="text-purple-600 text-xs" />
                      </div>
                      Email
                    </label>
                    <div className="w-full px-4 py-3 border border-gray-200 rounded-xl bg-gray-50/70 backdrop-blur-sm">
                      <p className="text-gray-900 font-medium">{profileData?.email || 'Not provided'}</p>
                    </div>
                  </div>

                  <div className="space-y-2">
                    <label className="flex items-center text-sm font-semibold text-gray-700 mb-3">
                      <div className="w-5 h-5 bg-indigo-100 rounded-lg flex items-center justify-center mr-3">
                        <FiBriefcase className="text-indigo-600 text-xs" />
                      </div>
                      Position
                    </label>
                    <div className="w-full px-4 py-3 border border-gray-200 rounded-xl bg-gray-50/70 backdrop-blur-sm">
                      <p className="text-gray-900 font-medium">{profileData?.position || 'Not provided'}</p>
                    </div>
                  </div>

                  <div className="space-y-2">
                    <label className="flex items-center text-sm font-semibold text-gray-700 mb-3">
                      <div className="w-5 h-5 bg-emerald-100 rounded-lg flex items-center justify-center mr-3">
                        <FiUsers className="text-emerald-600 text-xs" />
                      </div>
                      Department
                    </label>
                    <div className="w-full px-4 py-3 border border-gray-200 rounded-xl bg-gray-50/70 backdrop-blur-sm">
                      <p className="text-gray-900 font-medium">{profileData?.department || 'Not provided'}</p>
                    </div>
                  </div>
                </div>
              </div>

              {/* Company Information Section */}
              <div className="border-t border-gray-200 pt-10">
                <div className="flex items-center mb-8">
                  <div className="w-12 h-12 bg-gradient-to-r from-purple-500 to-pink-500 rounded-xl flex items-center justify-center mr-4">
                    <FiBriefcase className="text-white text-xl" />
                  </div>
                  <h2 className="text-2xl font-bold text-gray-900">Company Information</h2>
                </div>
                
                {/* Company Logo */}
                {profileData?.company?.logo_url && (
                  <div className="mb-8 flex justify-center">
                    <div className="relative">
                      <img
                        src={profileData.company.logo_url}
                        alt="Company Logo"
                        className="w-32 h-32 object-cover rounded-2xl border-4 border-white shadow-xl"
                      />
                      <div className="absolute -inset-1 bg-gradient-to-r from-purple-500 to-pink-500 rounded-2xl blur opacity-20"></div>
                    </div>
                  </div>
                )}

                <div className="grid md:grid-cols-2 gap-6">
                  <div className="space-y-2 md:col-span-2">
                    <label className="flex items-center text-sm font-semibold text-gray-700 mb-3">
                      <div className="w-5 h-5 bg-purple-100 rounded-lg flex items-center justify-center mr-3">
                        <FiFileText className="text-purple-600 text-xs" />
                      </div>
                      Company Name
                    </label>
                    <div className="w-full px-4 py-3 border border-gray-200 rounded-xl bg-gray-50/70 backdrop-blur-sm">
                      <p className="text-gray-900 font-medium">{profileData?.company?.name || 'Not provided'}</p>
                    </div>
                  </div>

                  {profileData?.company?.description && (
                    <div className="space-y-2 md:col-span-2">
                      <label className="flex items-center text-sm font-semibold text-gray-700 mb-3">
                        <div className="w-5 h-5 bg-blue-100 rounded-lg flex items-center justify-center mr-3">
                          <FiFileText className="text-blue-600 text-xs" />
                        </div>
                        Description
                      </label>
                      <div className="w-full px-4 py-3 border border-gray-200 rounded-xl bg-gray-50/70 backdrop-blur-sm">
                        <p className="text-gray-900">{profileData.company.description}</p>
                      </div>
                    </div>
                  )}

                  {profileData?.company?.industry && (
                    <div className="space-y-2">
                      <label className="flex items-center text-sm font-semibold text-gray-700 mb-3">
                        <div className="w-5 h-5 bg-pink-100 rounded-lg flex items-center justify-center mr-3">
                          <FiMapPin className="text-pink-600 text-xs" />
                        </div>
                        Industry
                      </label>
                      <div className="w-full px-4 py-3 border border-gray-200 rounded-xl bg-gray-50/70 backdrop-blur-sm">
                        <p className="text-gray-900 font-medium">{profileData.company.industry}</p>
                      </div>
                    </div>
                  )}

                  {profileData?.company?.size && (
                    <div className="space-y-2">
                      <label className="flex items-center text-sm font-semibold text-gray-700 mb-3">
                        <div className="w-5 h-5 bg-indigo-100 rounded-lg flex items-center justify-center mr-3">
                          <FiUsers className="text-indigo-600 text-xs" />
                        </div>
                        Company Size
                      </label>
                      <div className="w-full px-4 py-3 border border-gray-200 rounded-xl bg-gray-50/70 backdrop-blur-sm">
                        <p className="text-gray-900 font-medium">{profileData.company.size}</p>
                      </div>
                    </div>
                  )}

                  {profileData?.company?.website && (
                    <div className="space-y-2 md:col-span-2">
                      <label className="flex items-center text-sm font-semibold text-gray-700 mb-3">
                        <div className="w-5 h-5 bg-cyan-100 rounded-lg flex items-center justify-center mr-3">
                          <FiGlobe className="text-cyan-600 text-xs" />
                        </div>
                        Website
                      </label>
                      <div className="w-full px-4 py-3 border border-gray-200 rounded-xl bg-gray-50/70 backdrop-blur-sm">
                        <a 
                          href={profileData.company.website} 
                          target="_blank" 
                          rel="noopener noreferrer"
                          className="text-blue-600 hover:text-blue-800 hover:underline font-medium"
                        >
                          {profileData.company.website}
                        </a>
                      </div>
                    </div>
                  )}
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default ProfileRecruiter;
