import React, { useState, useEffect } from 'react';
import { useParams } from 'react-router-dom';
import { 
  FiStar, 
  FiBriefcase, 
  FiBook, 
  FiCheckCircle,
  FiXCircle 
} from 'react-icons/fi';
import apiClient from '../apiClient';

interface PublicProfileData {
  id: string;
  name: string;
  email: string;
  phone?: string;
  linkedin_url?: string;
  photo_url?: string;
  university?: string;
  graduation_year?: number;
  degree?: string;
  bio?: string;
  experiences: Array<{
    id: string;
    company: string;
    position: string;
    start_date: string;
    end_date?: string;
    description?: string;
  }>;
  educations: Array<{
    id: string;
    institution: string;
    degree: string;
    field_of_study?: string;
    start_date: string;
    end_date?: string;
    grade?: string;
  }>;
  skills: Array<{
    id: string;
    name: string;
    level?: string;
  }>;
}

const PublicProfile: React.FC = () => {
  const { uuid } = useParams<{ uuid: string }>();
  const [profile, setProfile] = useState<PublicProfileData | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [showBanner, setShowBanner] = useState(false);
  const [noteText, setNoteText] = useState('');
  const [saving, setSaving] = useState(false);
  
  // Toast notification states
  const [notification, setNotification] = useState<{
    show: boolean;
    type: 'success' | 'error';
    message: string;
  }>({ show: false, type: 'success', message: '' });

  // Function to show notification
  const showNotification = (type: 'success' | 'error', message: string) => {
    setNotification({ show: true, type, message });
    setTimeout(() => {
      setNotification({ show: false, type: 'success', message: '' });
    }, 4000);
  };

  // Check if user is authenticated recruiter
  const getUser = () => {
    const user = localStorage.getItem('user');
    return user ? JSON.parse(user) : null;
  };

  const currentUser = getUser();
  const isRecruiterAuthenticated = currentUser && currentUser.role === 'recruiter' && localStorage.getItem('token');

  useEffect(() => {
    const fetchPublicProfile = async () => {
      if (!uuid) {
        setError('Invalid profile URL');
        setLoading(false);
        return;
      }

      try {
        const response = await apiClient.get(`/api/p/${uuid}`);
        console.log('PublicProfile API Response:', response);
        console.log('Student data:', response.data.student);
        console.log('Skills:', response.data.student.skills);
        console.log('Experiences:', response.data.student.experiences);
        console.log('Educations:', response.data.student.educations);
        setProfile(response.data.student);
        
        // Show banner if recruiter is authenticated
        if (isRecruiterAuthenticated) {
          setShowBanner(true);
        }
      } catch (err: unknown) {
        console.error('Error fetching public profile:', err);
        const error = err as { response?: { status?: number } };
        if (error.response?.status === 404) {
          setError('Profile not found or inactive');
        } else {
          setError('Failed to load profile. Please try again.');
        }
      } finally {
        setLoading(false);
      }
    };

    fetchPublicProfile();
  }, [uuid, isRecruiterAuthenticated]);

  const handleSaveToShortlist = async () => {
    if (!profile || !isRecruiterAuthenticated) return;

    setSaving(true);
    try {
      await apiClient.post('/api/shortlist', {
        student_id: profile.id,
        notes: noteText || ''
      });
      
      setShowBanner(false);
      showNotification('success', 'Candidate successfully added to your shortlist!');
    } catch (err: unknown) {
      console.error('Error saving to shortlist:', err);
      const error = err as { response?: { status?: number } };
      if (error.response?.status === 409) {
        showNotification('error', 'This candidate is already in your shortlist');
      } else {
        showNotification('error', 'Failed to save candidate. Please try again.');
      }
    } finally {
      setSaving(false);
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
          <p className="mt-4 text-gray-600">Loading profile...</p>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="text-red-500 text-6xl mb-4">⚠️</div>
          <h1 className="text-2xl font-bold text-gray-900 mb-2">Profile Not Available</h1>
          <p className="text-gray-600 mb-4">{error}</p>
          <button 
            onClick={() => window.location.href = '/'}
            className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700"
          >
            Go to Homepage
          </button>
        </div>
      </div>
    );
  }

  if (!profile) {
    console.log('Profile is null or undefined:', profile);
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <p className="text-gray-600">Profile not found</p>
        </div>
      </div>
    );
  }

  console.log('Profile data before render:', profile);

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Toast Notification */}
      {notification.show && (
        <div className={`fixed top-4 right-4 z-50 p-4 rounded-xl shadow-xl transition-all duration-300 ${
          notification.type === 'success' 
            ? 'bg-green-500 text-white border border-green-400' 
            : 'bg-red-500 text-white border border-red-400'
        }`}>
          <div className="flex items-center space-x-3">
            <div className="flex-shrink-0">
              {notification.type === 'success' ? (
                <FiCheckCircle className="text-lg" />
              ) : (
                <FiXCircle className="text-lg" />
              )}
            </div>
            <span className="font-medium flex-1">{notification.message}</span>
            <button
              onClick={() => setNotification({ show: false, type: 'success', message: '' })}
              className="ml-2 text-white hover:text-gray-200 transition-colors"
            >
              ✕
            </button>
          </div>
        </div>
      )}
      
      {/* Recruiter Banner */}
      {showBanner && (
        <div className="fixed top-0 left-0 right-0 bg-blue-600 text-white p-4 z-50 shadow-lg">
          <div className="max-w-4xl mx-auto flex items-center justify-between">
            <div className="flex items-center space-x-4">
              <span className="font-medium">💼 Recruiter Mode Active</span>
              <input
                type="text"
                placeholder="Add a private note (optional)"
                value={noteText}
                onChange={(e) => setNoteText(e.target.value)}
                className="px-3 py-1 text-black rounded border-0 focus:ring-2 focus:ring-blue-300"
              />
            </div>
            <div className="flex items-center space-x-3">
              <button
                onClick={handleSaveToShortlist}
                disabled={saving}
                className="bg-green-500 hover:bg-green-600 px-4 py-2 rounded font-medium disabled:opacity-50"
              >
                {saving ? 'Saving...' : 'Save to Shortlist'}
              </button>
              <button
                onClick={() => setShowBanner(false)}
                className="text-white hover:text-gray-200"
              >
                ✕
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Profile Content */}
      <div className={`max-w-4xl mx-auto p-6 ${showBanner ? 'pt-20' : ''}`}>
        {/* Header Section */}
        <div className="bg-white rounded-lg shadow-md p-6 mb-6">
          <div className="flex flex-col md:flex-row items-center md:items-start space-y-4 md:space-y-0 md:space-x-6">
            <div className="flex-shrink-0">
              {profile.photo_url ? (
                <img
                  src={profile.photo_url}
                  alt={profile.name}
                  className="w-32 h-32 rounded-full object-cover border-4 border-gray-200"
                />
              ) : (
                <div className="w-32 h-32 rounded-full bg-gray-300 flex items-center justify-center text-gray-600 text-4xl font-bold">
                  {profile.name.charAt(0).toUpperCase()}
                </div>
              )}
            </div>
            
            <div className="flex-1 text-center md:text-left">
              <h1 className="text-3xl font-bold text-gray-900 mb-2">{profile.name}</h1>
              {profile.degree && profile.university && (
                <p className="text-xl text-gray-600 mb-3">
                  {profile.degree} - {profile.university}
                  {profile.graduation_year && ` (${profile.graduation_year})`}
                </p>
              )}
              
              <div className="flex flex-wrap gap-4 justify-center md:justify-start text-sm text-gray-600">
                {profile.email && (
                  <a href={`mailto:${profile.email}`} className="flex items-center hover:text-blue-600">
                    <span className="mr-1">📧</span>
                    {profile.email}
                  </a>
                )}
                {profile.phone && (
                  <a href={`tel:${profile.phone}`} className="flex items-center hover:text-blue-600">
                    <span className="mr-1">📱</span>
                    {profile.phone}
                  </a>
                )}
                {profile.linkedin_url && (
                  <a href={profile.linkedin_url} target="_blank" rel="noopener noreferrer" className="flex items-center hover:text-blue-600">
                    <span className="mr-1">💼</span>
                    LinkedIn
                  </a>
                )}
              </div>
              
              {profile.bio && (
                <p className="mt-4 text-gray-700 leading-relaxed">{profile.bio}</p>
              )}
            </div>
          </div>
        </div>

        {/* Skills Section */}
        {profile.skills && profile.skills.length > 0 && (
          <div className="bg-white rounded-lg shadow-md p-6 mb-6">
            <div className="flex items-center mb-4">
              <FiStar className="text-gray-700 text-2xl mr-2" />
              <h2 className="text-2xl font-bold text-gray-900">Skills</h2>
            </div>
            <div className="flex flex-wrap gap-2">
              {profile.skills.map((skill) => (
                <span
                  key={skill.id}
                  className="bg-blue-100 text-blue-800 px-3 py-1 rounded-full text-sm font-medium"
                >
                  {skill.name}
                  {skill.level && ` (${skill.level})`}
                </span>
              ))}
            </div>
          </div>
        )}

        {/* Experience Section */}
        {profile.experiences && profile.experiences.length > 0 && (
          <div className="bg-white rounded-lg shadow-md p-6 mb-6">
            <div className="flex items-center mb-4">
              <FiBriefcase className="text-gray-700 text-2xl mr-2" />
              <h2 className="text-2xl font-bold text-gray-900">Professional Experience</h2>
            </div>
            <div className="space-y-4">
              {profile.experiences.map((exp) => (
                <div key={exp.id} className="border-l-4 border-blue-500 pl-4">
                  <h3 className="text-lg font-semibold text-gray-900">{exp.position}</h3>
                  <p className="text-gray-700 font-medium">{exp.company}</p>
                  <p className="text-sm text-gray-600 mb-2">
                    {exp.start_date || 'Unknown'} - 
                    {exp.end_date || 'Present'}
                  </p>
                  {exp.description && (
                    <p className="text-gray-700">{exp.description}</p>
                  )}
                </div>
              ))}
            </div>
          </div>
        )}

        {/* Education Section */}
        {profile.educations && profile.educations.length > 0 && (
          <div className="bg-white rounded-lg shadow-md p-6 mb-6">
            <div className="flex items-center mb-4">
              <FiBook className="text-gray-700 text-2xl mr-2" />
              <h2 className="text-2xl font-bold text-gray-900">Education</h2>
            </div>
            <div className="space-y-4">
              {profile.educations.map((edu) => (
                <div key={edu.id} className="border-l-4 border-green-500 pl-4">
                  <h3 className="text-lg font-semibold text-gray-900">{edu.degree}</h3>
                  {edu.field_of_study && (
                    <p className="text-gray-700">{edu.field_of_study}</p>
                  )}
                  <p className="text-gray-700 font-medium">{edu.institution}</p>
                  <p className="text-sm text-gray-600 mb-2">
                    {edu.start_date || 'Unknown'} - 
                    {edu.end_date || 'Present'}
                  </p>
                  {edu.grade && (
                    <p className="text-gray-700">Grade: {edu.grade}</p>
                  )}
                </div>
              ))}
            </div>
          </div>
        )}

        {/* Footer */}
        <div className="text-center py-8">
          <p className="text-gray-500 text-sm">
            Powered by <span className="font-semibold text-blue-600">PopCareers</span>
          </p>
        </div>
      </div>
    </div>
  );
};

export default PublicProfile;
