import React, { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import { 
  FiBriefcase, 
  FiUsers, 
  FiZap,
  FiSearch,
  FiUser,
  FiStar,
  FiTarget,
  FiTrendingUp
} from 'react-icons/fi';
import apiClient from '../apiClient';

interface Recruiter {
  id: string;
  name: string;
  email: string;
  company: {
    id: string;
    name: string;
    logo_url?: string;
  };
}

interface ShortlistCandidate {
  id: string;
  student: {
    id: string;
    name: string;
    email: string;
    photo_url?: string;
    university?: string;
    degree?: string;
    skills: Array<{ name: string }>;
  };
  notes?: string;
  created_at: string;
}

interface DashboardStats {
  total_shortlisted: number;
  recent_scans: number;
  active_jobs: number;
  profile_views: number;
}

interface ActivityLog {
  id: string;
  user_id: string;
  action: string;
  created_at: string;
}

const RecruiterDashboard: React.FC = () => {
  const [recruiter, setRecruiter] = useState<Recruiter | null>(null);
  const [shortlist, setShortlist] = useState<ShortlistCandidate[]>([]);
  const [stats, setStats] = useState<DashboardStats | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [searchTerm, setSearchTerm] = useState('');

  useEffect(() => {
    fetchDashboardData();
  }, []);

  const fetchDashboardData = async () => {
    try {
      const [profileResponse, shortlistResponse, activityResponse] = await Promise.all([
        apiClient.get('/api/user'),
        apiClient.get('/api/shortlist'),
        apiClient.get('/api/activity-logs').catch(() => ({ data: { logs: [] } }))
      ]);

      const userData = profileResponse.data.data?.user || profileResponse.data.user || profileResponse.data;
      
      // Map user data to recruiter format using correct data structure
      const recruiterData = {
        id: userData.id,
        name: userData.profile?.full_name || userData.name || userData.email?.split('@')[0] || 'Recruiter',
        email: userData.email,
        company: {
          id: userData.profile?.company?.id || userData.id,
          name: userData.profile?.company_name || userData.profile?.company?.name || 'Your Company',
          logo_url: userData.profile?.photo_url || null
        }
      };
      
      const shortlistData = shortlistResponse.data.shortlist || shortlistResponse.data || [];
      const activityLogs: ActivityLog[] = activityResponse.data?.logs || activityResponse.data || [];
      
      // Calculate live stats
      const now = new Date();
      const last7Days = new Date(now.getTime() - 7 * 24 * 60 * 60 * 1000);
      const last30Days = new Date(now.getTime() - 30 * 24 * 60 * 60 * 1000);
      
      // Count recent scanning activities (last 7 days) - when THIS recruiter scanned QR codes
      const recentScans = activityLogs.filter((log: ActivityLog) => {
        if (!log.created_at) return false;
        const logDate = new Date(log.created_at);
        return logDate >= last7Days && 
               log.user_id === userData.id && 
               log.action && (
                 log.action.toLowerCase().includes('scan') || 
                 log.action.toLowerCase().includes('qr') ||
                 log.action.toLowerCase().includes('scanned_qr')
               );
      }).length;
      
      // Count profile views by this recruiter in last 30 days
      const profileViews = activityLogs.filter((log: ActivityLog) => {
        if (!log.created_at) return false;
        const logDate = new Date(log.created_at);
        return logDate >= last30Days && 
               log.user_id === userData.id &&
               log.action && (
                 log.action.toLowerCase().includes('view') || 
                 log.action.toLowerCase().includes('profile')
               );
      }).length;
      
      // Count shortlist additions in last 7 days
      const recentShortlistAdditions = shortlistData.filter((item: ShortlistCandidate) => {
        if (!item.created_at) return false;
        const itemDate = new Date(item.created_at);
        return itemDate >= last7Days;
      }).length;
      
      console.log('Dashboard stats calculation:', {
        totalShortlisted: shortlistData.length,
        recentScans,
        profileViews,
        recentShortlistAdditions,
        activityLogsCount: activityLogs.length
      });
      
      setRecruiter(recruiterData);
      setShortlist(shortlistData);
      setStats({
        total_shortlisted: shortlistData.length,
        recent_scans: recentScans,
        active_jobs: recentShortlistAdditions, // Use recent shortlist additions as active jobs for now
        profile_views: profileViews
      });
    } catch (err) {
      console.error('Error fetching dashboard data:', err);
      setError('Failed to load dashboard');
    } finally {
      setLoading(false);
    }
  };

  const removeFromShortlist = async (shortlistId: string) => {
    if (!confirm('Are you sure you want to remove this candidate from your shortlist?')) {
      return;
    }

    try {
      await apiClient.delete(`/api/shortlist/${shortlistId}`);
      setShortlist(prev => prev.filter(item => item.id !== shortlistId));
      
      if (stats) {
        setStats(prev => prev ? { ...prev, total_shortlisted: prev.total_shortlisted - 1 } : null);
      }
    } catch (err) {
      console.error('Error removing from shortlist:', err);
      alert('Failed to remove candidate from shortlist');
    }
  };

  const filteredShortlist = shortlist.filter(item =>
    item?.student?.name?.toLowerCase().includes(searchTerm.toLowerCase()) ||
    item?.student?.university?.toLowerCase().includes(searchTerm.toLowerCase()) ||
    item?.student?.skills?.some(skill => skill?.name?.toLowerCase().includes(searchTerm.toLowerCase()))
  );

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
          <p className="mt-4 text-gray-600">Loading your dashboard...</p>
        </div>
      </div>
    );
  }

  if (error || !recruiter) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="text-red-500 text-6xl mb-4">⚠️</div>
          <h1 className="text-2xl font-bold text-gray-900 mb-2">Dashboard Error</h1>
          <p className="text-gray-600 mb-4">{error || 'Failed to load dashboard'}</p>
          <button 
            onClick={fetchDashboardData}
            className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700"
          >
            Try Again
          </button>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-100">
      {/* Hero Section */}
      <div className="relative overflow-hidden bg-gradient-to-r from-blue-600 via-purple-600 to-indigo-700 text-white">
        <div className="absolute inset-0 bg-black/10"></div>
        <div className="absolute inset-0 opacity-20">
          <div 
            className="w-full h-full"
            style={{
              backgroundImage: `url("data:image/svg+xml,%3Csvg width='60' height='60' viewBox='0 0 60 60' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='none' fill-rule='evenodd'%3E%3Cg fill='%23ffffff' fill-opacity='0.05'%3E%3Ccircle cx='30' cy='30' r='4'/%3E%3C/g%3E%3C/g%3E%3C/svg%3E")`
            }}
          ></div>
        </div>
        
        <div className="relative max-w-7xl mx-auto px-6 py-16">
          <div className="flex items-center justify-between">
            <div className="flex items-center">
              <div className="relative">
                <div className="w-20 h-20 bg-white/20 backdrop-blur-sm rounded-2xl flex items-center justify-center mr-6 border border-white/30">
                  <FiBriefcase className="text-white text-3xl" />
                </div>
                <div className="absolute -inset-1 bg-gradient-to-r from-white/20 to-white/5 rounded-2xl blur-lg"></div>
              </div>
              <div>
                <h1 className="text-5xl font-bold mb-3 bg-gradient-to-r from-white to-blue-100 bg-clip-text text-transparent">
                  Welcome back, {recruiter?.name || 'Recruiter'}!
                </h1>
                <p className="text-xl text-blue-100 flex items-center mb-2">
                  <FiBriefcase className="mr-3 text-lg" />
                  {recruiter?.company?.name || 'Your Company'} • Talent Acquisition Hub
                </p>
                <p className="text-blue-200">
                  Discover, connect, and build your dream team with cutting-edge recruitment tools 🚀
                </p>
              </div>
            </div>
            
            {/* Company Logo */}
            {recruiter?.company?.logo_url && (
              <div className="relative">
                <div className="w-24 h-24 bg-white/10 backdrop-blur-sm rounded-2xl p-4 border border-white/20">
                  <img
                    src={recruiter.company.logo_url}
                    alt={recruiter.company.name || 'Company logo'}
                    className="w-full h-full object-contain"
                  />
                </div>
                <div className="absolute -inset-1 bg-gradient-to-r from-white/10 to-transparent rounded-2xl blur-lg"></div>
              </div>
            )}
          </div>
          
          {/* Quick Action Buttons */}
          <div className="flex space-x-4 mt-8">
            <Link
              to="/scan"
              className="group relative overflow-hidden bg-white/10 backdrop-blur-sm border border-white/20 text-white px-8 py-4 rounded-xl hover:bg-white/20 transition-all duration-300 flex items-center space-x-3"
            >
              <FiZap className="text-xl" />
              <span className="font-semibold">Scan QR Code</span>
            </Link>
            <Link
              to="/shortlist"
              className="group relative overflow-hidden bg-gradient-to-r from-orange-500 to-red-500 text-white px-8 py-4 rounded-xl hover:from-orange-600 hover:to-red-600 transition-all duration-300 shadow-lg hover:shadow-xl flex items-center space-x-3"
            >
              <FiStar className="text-xl" />
              <span className="font-semibold">View Shortlist</span>
            </Link>
          </div>
        </div>
      </div>

      {/* Main Content */}
      <div className="max-w-7xl mx-auto px-6 py-12">
        {/* Modern Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-12">
          {/* Total Scans */}
          <div className="group relative overflow-hidden bg-white/70 backdrop-blur-sm border border-white/50 rounded-2xl p-6 hover:bg-white/80 transition-all duration-300 shadow-lg hover:shadow-xl">
            <div className="absolute top-0 right-0 w-32 h-32 bg-gradient-to-br from-blue-500/10 to-purple-500/10 rounded-full blur-xl transform translate-x-16 -translate-y-16"></div>
            <div className="relative">
              <div className="flex items-center justify-between mb-4">
                <div className="w-12 h-12 bg-gradient-to-br from-blue-500 to-blue-600 rounded-xl flex items-center justify-center shadow-lg">
                  <FiZap className="text-white text-xl" />
                </div>
                <span className="text-2xl font-bold text-gray-900">{stats?.recent_scans || 0}</span>
              </div>
              <h3 className="text-lg font-bold text-gray-900 mb-1">Total Scans</h3>
              <p className="text-sm text-gray-600">QR codes scanned</p>
            </div>
          </div>

          {/* Profile Views */}
          <div className="group relative overflow-hidden bg-white/70 backdrop-blur-sm border border-white/50 rounded-2xl p-6 hover:bg-white/80 transition-all duration-300 shadow-lg hover:shadow-xl">
            <div className="absolute top-0 right-0 w-32 h-32 bg-gradient-to-br from-green-500/10 to-emerald-500/10 rounded-full blur-xl transform translate-x-16 -translate-y-16"></div>
            <div className="relative">
              <div className="flex items-center justify-between mb-4">
                <div className="w-12 h-12 bg-gradient-to-br from-green-500 to-green-600 rounded-xl flex items-center justify-center shadow-lg">
                  <FiUsers className="text-white text-xl" />
                </div>
                <span className="text-2xl font-bold text-gray-900">{stats?.profile_views || 0}</span>
              </div>
              <h3 className="text-lg font-bold text-gray-900 mb-1">Profile Views</h3>
              <p className="text-sm text-gray-600">Candidates viewed</p>
            </div>
          </div>

          {/* Shortlisted */}
          <div className="group relative overflow-hidden bg-white/70 backdrop-blur-sm border border-white/50 rounded-2xl p-6 hover:bg-white/80 transition-all duration-300 shadow-lg hover:shadow-xl">
            <div className="absolute top-0 right-0 w-32 h-32 bg-gradient-to-br from-orange-500/10 to-red-500/10 rounded-full blur-xl transform translate-x-16 -translate-y-16"></div>
            <div className="relative">
              <div className="flex items-center justify-between mb-4">
                <div className="w-12 h-12 bg-gradient-to-br from-orange-500 to-red-500 rounded-xl flex items-center justify-center shadow-lg">
                  <FiStar className="text-white text-xl" />
                </div>
                <span className="text-2xl font-bold text-gray-900">{stats?.total_shortlisted || 0}</span>
              </div>
              <h3 className="text-lg font-bold text-gray-900 mb-1">Shortlisted</h3>
              <p className="text-sm text-gray-600">Top candidates</p>
            </div>
          </div>

          {/* Conversion Rate */}
          <div className="group relative overflow-hidden bg-white/70 backdrop-blur-sm border border-white/50 rounded-2xl p-6 hover:bg-white/80 transition-all duration-300 shadow-lg hover:shadow-xl">
            <div className="absolute top-0 right-0 w-32 h-32 bg-gradient-to-br from-purple-500/10 to-indigo-500/10 rounded-full blur-xl transform translate-x-16 -translate-y-16"></div>
            <div className="relative">
              <div className="flex items-center justify-between mb-4">
                <div className="w-12 h-12 bg-gradient-to-br from-purple-500 to-indigo-500 rounded-xl flex items-center justify-center shadow-lg">
                  <FiTrendingUp className="text-white text-xl" />
                </div>
                <span className="text-2xl font-bold text-gray-900">{stats?.recent_scans || 0}</span>
              </div>
              <h3 className="text-lg font-bold text-gray-900 mb-1">Recent Activity</h3>
              <p className="text-sm text-gray-600">Scans (7 days)</p>
            </div>
          </div>
        </div>

        {/* Enhanced Content Grid */}
        <div className="grid lg:grid-cols-4 gap-8">
          {/* Main Content */}
          <div className="lg:col-span-3">
            {/* Quick Actions */}
            <div className="bg-white rounded-xl shadow-md p-6 mb-8 border border-gray-100">
              <div className="flex items-center mb-4">
                <FiTarget className="text-gray-700 text-xl mr-2" />
                <h2 className="text-xl font-semibold text-gray-900">Quick Actions</h2>
              </div>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                <Link
                  to="/scanner"
                  className="flex items-center p-4 bg-blue-50 rounded-xl hover:bg-blue-100 transition-colors border border-blue-100"
                >
                  <div className="w-10 h-10 bg-blue-500 rounded-lg flex items-center justify-center mr-3">
                    <FiSearch className="text-white text-lg" />
                  </div>
                  <div>
                    <div className="font-medium text-gray-900">QR Scanner</div>
                    <div className="text-sm text-gray-600">Scan student profiles</div>
                  </div>
                </Link>
                
                <Link
                  to="/shortlist"
                  className="flex items-center p-4 bg-green-50 rounded-xl hover:bg-green-100 transition-colors border border-green-100"
                >
                  <div className="w-10 h-10 bg-green-500 rounded-lg flex items-center justify-center mr-3">
                    <FiStar className="text-white text-lg" />
                  </div>
                  <div>
                    <div className="font-medium text-gray-900">Full Shortlist</div>
                    <div className="text-sm text-gray-600">Manage all candidates</div>
                  </div>
                </Link>
                
                <Link
                  to="/profile"
                  className="flex items-center p-4 bg-purple-50 rounded-xl hover:bg-purple-100 transition-colors border border-purple-100"
                >
                  <div className="w-10 h-10 bg-purple-500 rounded-lg flex items-center justify-center mr-3">
                    <FiUser className="text-white text-lg" />
                  </div>
                  <div>
                    <div className="font-medium text-gray-900">Edit Profile</div>
                    <div className="text-sm text-gray-600">Update your info</div>
                  </div>
                </Link>
              </div>
            </div>

            {/* Shortlist Preview */}
            <div className="bg-white rounded-xl shadow-md p-6 border border-gray-100">
              <div className="flex items-center justify-between mb-6">
                <div className="flex items-center">
                  <FiStar className="text-gray-700 text-xl mr-2" />
                  <h2 className="text-xl font-semibold text-gray-900">Recent Shortlist</h2>
                </div>
                <Link
                  to="/shortlist"
                  className="text-blue-600 hover:text-blue-700 font-medium"
                >
                  View All →
                </Link>
              </div>

              {/* Search */}
              <div className="mb-4">
                <input
                  type="text"
                  placeholder="Search candidates by name, university, or skills..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>

              {filteredShortlist.length === 0 ? (
                <div className="text-center py-8">
                  {shortlist.length === 0 ? (
                    <div>
                      <div className="text-gray-400 text-6xl mb-4">📝</div>
                      <h3 className="text-lg font-medium text-gray-900 mb-2">No candidates yet</h3>
                      <p className="text-gray-600 mb-4">Start scanning QR codes at events to build your shortlist</p>
                      <Link
                        to="/scanner"
                        className="inline-block bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700"
                      >
                        Start Scanning
                      </Link>
                    </div>
                  ) : (
                    <div>
                      <div className="text-gray-400 text-4xl mb-2">🔍</div>
                      <p className="text-gray-600">No candidates match your search</p>
                    </div>
                  )}
                </div>
              ) : (
                <div className="space-y-4">
                  {filteredShortlist.slice(0, 5).map((item) => (
                    <div key={item.id} className="border border-gray-200 rounded-lg p-4 hover:bg-gray-50 transition-colors">
                      <div className="flex items-center justify-between">
                        <div className="flex items-center space-x-4">
                          {item.student.photo_url ? (
                            <img
                              src={item.student.photo_url}
                              alt={item.student.name}
                              className="w-12 h-12 rounded-full object-cover"
                            />
                          ) : (
                            <div className="w-12 h-12 rounded-full bg-gray-300 flex items-center justify-center text-gray-600 font-bold">
                              {item.student.name.charAt(0).toUpperCase()}
                            </div>
                          )}
                          
                          <div className="flex-1">
                            <h3 className="font-medium text-gray-900">{item.student.name}</h3>
                            {item.student.degree && item.student.university && (
                              <p className="text-sm text-gray-600">
                                {item.student.degree} - {item.student.university}
                              </p>
                            )}
                            <p className="text-xs text-gray-500">{item.student.email}</p>
                            
                            {/* Top Skills */}
                            {item.student.skills.length > 0 && (
                              <div className="flex flex-wrap gap-1 mt-2">
                                {item.student.skills.slice(0, 3).map((skill, index) => (
                                  <span
                                    key={index}
                                    className="bg-blue-100 text-blue-800 px-2 py-1 rounded-full text-xs"
                                  >
                                    {skill.name}
                                  </span>
                                ))}
                                {item.student.skills.length > 3 && (
                                  <span className="text-xs text-gray-500">
                                    +{item.student.skills.length - 3} more
                                  </span>
                                )}
                              </div>
                            )}
                            
                            {/* Notes */}
                            {item.notes && (
                              <p className="text-sm text-gray-700 mt-2 italic">
                                Note: {item.notes}
                              </p>
                            )}
                          </div>
                        </div>
                        
                        <div className="flex items-center space-x-2">
                          <a
                            href={`/p/${item.student.id}`}
                            target="_blank"
                            rel="noopener noreferrer"
                            className="bg-blue-600 text-white px-3 py-1 rounded text-sm hover:bg-blue-700 transition-colors"
                          >
                            View Profile
                          </a>
                          <button
                            onClick={() => removeFromShortlist(item.id)}
                            className="bg-red-600 text-white px-3 py-1 rounded text-sm hover:bg-red-700 transition-colors"
                          >
                            Remove
                          </button>
                        </div>
                      </div>
                      
                      <div className="mt-2 text-xs text-gray-500">
                        Added {new Date(item.created_at).toLocaleDateString()}
                      </div>
                    </div>
                  ))}
                  
                  {filteredShortlist.length > 5 && (
                    <div className="text-center pt-4">
                      <Link
                        to="/shortlist"
                        className="text-blue-600 hover:text-blue-700 font-medium"
                      >
                        View {filteredShortlist.length - 5} more candidates →
                      </Link>
                    </div>
                  )}
                </div>
              )}
            </div>
          </div>

          {/* Sidebar */}
          <div className="space-y-6">
            {/* Company Info */}
            <div className="bg-white rounded-xl shadow-md p-6 border border-gray-100">
              <div className="flex items-center mb-4">
                <FiBriefcase className="text-gray-700 text-xl mr-2" />
                <h3 className="text-lg font-semibold text-gray-900">Company</h3>
              </div>
              <div className="text-center">
                {recruiter.company?.logo_url ? (
                  <img
                    src={recruiter.company.logo_url}
                    alt={recruiter.company.name || 'Company logo'}
                    className="w-16 h-16 object-contain mx-auto mb-3 rounded-lg"
                  />
                ) : (
                  <div className="w-16 h-16 bg-gradient-to-r from-blue-500 to-purple-600 mx-auto mb-3 rounded-lg flex items-center justify-center text-white text-xl">
                    <FiBriefcase />
                  </div>
                )}
                <h4 className="font-medium text-gray-900">{recruiter.company?.name || 'Your Company'}</h4>
                <p className="text-sm text-gray-600 mt-1">{recruiter.name || 'Recruiter'}</p>
                <p className="text-xs text-gray-500">{recruiter.email || ''}</p>
              </div>
            </div>

            {/* Scanner Access */}
            <div className="bg-gradient-to-r from-blue-50 to-indigo-50 rounded-xl p-6 border border-blue-100">
              <div className="flex items-center mb-3">
                <FiSearch className="text-blue-600 text-xl mr-2" />
                <h3 className="text-lg font-semibold text-blue-900">Quick Scanner</h3>
              </div>
              <p className="text-sm text-blue-800 mb-4">
                Scan student QR codes at events to instantly view profiles and add to shortlist
              </p>
              <Link
                to="/scanner"
                className="block w-full text-center bg-blue-600 text-white py-3 rounded-lg hover:bg-blue-700 transition-colors font-medium"
              >
                Open Scanner
              </Link>
            </div>

            {/* Tips */}
            <div className="bg-green-50 rounded-lg p-6">
              <h3 className="text-lg font-semibold text-green-900 mb-3">💡 Recruiting Tips</h3>
              <ul className="space-y-2 text-sm text-green-800">
                <li>• Use the mobile scanner at events</li>
                <li>• Add notes immediately after scanning</li>
                <li>• Review shortlist candidates regularly</li>
                <li>• Share profile links with your team</li>
                <li>• Follow up with promising candidates</li>
              </ul>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default RecruiterDashboard;