import React, { useState, useEffect, useCallback } from 'react';
import { Link } from 'react-router-dom';
import { 
  FiUser, 
  FiEye, 
  FiBarChart,
  FiEdit3,
  FiCode,
  FiExternalLink,
  FiCheckCircle,
  FiXCircle,
  FiClock,
  FiTrendingUp,
  FiActivity,
  FiCalendar,
  FiTarget,
  FiBriefcase,
  FiBookOpen,
  FiAward,
  FiShare2,
  FiMapPin,
  FiFilter,
  FiMoreHorizontal,
  FiArrowUp,
  FiArrowDown,
  FiPieChart,
  FiZap,
  FiHeart
} from 'react-icons/fi';
import apiClient from '../apiClient';

interface Student {
  id: string;
  name: string;
  email: string;
  photo_url?: string;
  university?: string;
  degree?: string;
  graduation_year?: number;
  profile_completion: number;
  qr_active: boolean;
  experiences_count: number;
  educations_count: number;
  skills_count: number;
}

interface ActivityLog {
  id: string;
  user_id: string;
  action: string;
  created_at: string;
}

interface ProfileStats {
  profile_views: number;
  profile_views_change: number;
  qr_scans: number;
  qr_scans_change: number;
  shortlist_additions: number;
  shortlist_additions_change: number;
  recent_activity: ActivityLog[];
}

interface AiTip {
  title: string;
  description: string;
  action: string;
  category?: 'profile' | 'skills' | 'experience' | 'branding' | 'readiness';
  priority?: 'high' | 'medium' | 'low';
  impact_score?: number;
  metric?: string;
}

const StudentDashboard: React.FC = () => {
  const [student, setStudent] = useState<Student | null>(null);
  const [stats, setStats] = useState<ProfileStats | null>(null);
  const [proTips, setProTips] = useState<AiTip[]>([]);
  const [tipsStatus, setTipsStatus] = useState<'idle' | 'loading' | 'success' | 'fallback' | 'error'>('idle');
  const [tipsMessage, setTipsMessage] = useState<string | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  interface UserProfile {
    full_name?: string;
    phone?: string;
    summary?: string;
    linkedin?: string;
    photo_url?: string;
  }

  interface UserData {
    profile?: UserProfile;
  }

  const fetchDashboardData = useCallback(async () => {
    const calculateProfileCompletion = (userData: UserData, experiences: unknown[] = [], skills: unknown[] = [], photoUrl: string | null = null) => {
      const criteria = [
        userData.profile?.full_name,
        userData.profile?.phone,
        userData.profile?.summary,
        userData.profile?.linkedin,
        experiences.length > 0,
        skills.length > 0,
        photoUrl || userData.profile?.photo_url
      ];
      
      const completedCriteria = criteria.filter(Boolean).length;
      return Math.round((completedCriteria / criteria.length) * 100);
    };

    setTipsStatus('loading');
    setTipsMessage(null);

    try {
      console.log('Fetching dashboard data...');
      
      // Get user profile data
      const profileResponse = await apiClient.get('/api/user');
      console.log('Profile response:', profileResponse.data);
      
      // Get dashboard statistics
      let statsResponse = { 
        data: { 
          stats: { 
            profile_views: 0, 
            profile_views_change: 0,
            qr_scans: 0, 
            qr_scans_change: 0,
            shortlist_additions: 0,
            shortlist_additions_change: 0,
            recent_activity: [] 
          } 
        } 
      };
      try {
        statsResponse = await apiClient.get('/api/dashboard-stats');
        console.log('Dashboard stats response:', statsResponse.data);
      } catch (statsError) {
        console.log('Dashboard stats not available:', statsError);
      }

      // Extract user data from the correct structure
      const userData = profileResponse.data.data?.user || profileResponse.data.user;
      console.log('User data structure:', userData);
      
      if (!userData) {
        throw new Error('No user data received from API');
      }

      // Map the user data to student interface using correct structure
      const experiences = userData.profile?.experiences || [];
      const educations = userData.profile?.educations || [];
      const skills = userData.profile?.skills || [];
      const photoUrl = userData.profile?.photo_url;
      
      // Calculate consistent profile completion
      const profileCompletion = calculateProfileCompletion(userData, experiences, skills, photoUrl);
      
      const studentData: Student = {
        id: userData.id,
        name: userData.profile?.full_name || userData.name || userData.email?.split('@')[0] || 'Student',
        email: userData.email,
        photo_url: photoUrl,
        university: userData.profile?.university,
        degree: userData.profile?.degree,
        graduation_year: userData.profile?.graduation_year,
        profile_completion: profileCompletion,
        qr_active: userData.profile?.is_card_active || false,
        experiences_count: experiences.length,
        educations_count: educations.length,
        skills_count: skills.length,
      };

      console.log('Mapped student data:', studentData);

      // Use stats from the dedicated dashboard-stats endpoint
      const rawStats = statsResponse.data?.stats || {};
      const dashboardStats: ProfileStats = {
        profile_views: rawStats['profile_views'] || 0,
        profile_views_change: rawStats['profile_views_change'] || 0,
        qr_scans: rawStats['qr_scans'] || 0,
        qr_scans_change: rawStats['qr_scans_change'] || 0,
        shortlist_additions: rawStats['shortlist_additions'] || 0,
        shortlist_additions_change: rawStats['shortlist_additions_change'] || 0,
        recent_activity: rawStats['recent_activity'] || []
      };

      console.log('Dashboard stats:', dashboardStats);

      try {
        const tipsResponse = await apiClient.get('/api/student/pro-tips');
        const tipsPayload = tipsResponse.data;
        const aiTips = tipsPayload?.tips || [];

        if (aiTips.length === 0) {
          setProTips(generateLocalTips(studentData));
          setTipsStatus('fallback');
          setTipsMessage('AI returned no tips, using local suggestions.');
        } else {
          setProTips(aiTips);
          const isFallback = tipsPayload?.status === 'fallback';
          setTipsStatus(isFallback ? 'fallback' : 'success');
          if (isFallback && tipsPayload?.reason) {
            setTipsMessage(`Using fallback tips: ${tipsPayload.reason}`);
          }
        }
      } catch (tipsError) {
        console.error('Error fetching pro tips:', tipsError);
        setProTips(generateLocalTips(studentData));
        setTipsStatus('error');
        setTipsMessage('Using local suggestions while AI tips are unavailable.');
      }

      setStudent(studentData);
      setStats(dashboardStats);
    } catch (err: unknown) {
      console.error('Error fetching dashboard data:', err);
      const error = err as { response?: { data?: { message?: string } }; message?: string };
      console.error('Error details:', error.response?.data || error.message);
      setError(`Failed to load dashboard: ${error.response?.data?.message || error.message || 'Unknown error'}`);
      setTipsStatus('error');
    } finally {
      setLoading(false);
    }
  }, []);

  useEffect(() => {
    fetchDashboardData();
  }, [fetchDashboardData]);

  interface UserProfile {
    full_name?: string;
    phone?: string;
    summary?: string;
    linkedin?: string;
    photo_url?: string;
  }

  const getCompletionColor = (completion: number) => {
    if (completion >= 80) return 'text-green-600';
    if (completion >= 50) return 'text-yellow-600';
    return 'text-red-600';
  };

  const getCompletionBgColor = (completion: number) => {
    if (completion >= 80) return 'bg-green-100 border-green-200';
    if (completion >= 50) return 'bg-yellow-100 border-yellow-200';
    return 'bg-red-100 border-red-200';
  };

  const formatPercentageChange = (change: number) => {
    if (change === 0) return { text: '0%', color: 'text-gray-500', icon: null };
    if (change > 0) return { text: `+${change}%`, color: 'text-green-600', icon: 'up' };
    return { text: `${change}%`, color: 'text-red-600', icon: 'down' };
  };

  const generateLocalTips = (studentData: Student): AiTip[] => {
    const missingSummary = !studentData.name || !studentData.email;
    const needsSkills = studentData.skills_count < 5;
    const needsExperience = studentData.experiences_count < 2;

    return [
      {
        title: 'Sharpen your summary',
        description: 'Add a 2-line summary with your target role, stack, and one impact metric.',
        action: 'Open your profile and refresh the summary.',
        category: 'branding' as const,
        priority: 'high' as const,
        impact_score: 4,
        metric: 'profile_completion',
      },
      {
        title: 'Showcase measurable wins',
        description: 'Rewrite one experience with numbers (users reached, time saved, revenue impact).',
        action: 'Edit one experience entry today.',
        category: 'experience' as const,
        priority: (needsExperience ? 'high' : 'medium') as 'high' | 'medium',
        impact_score: needsExperience ? 5 : 3,
        metric: 'recruiter_interest',
      },
      {
        title: 'Label top skills',
        description: 'List 5-8 skills with levels to help recruiters filter you in searches.',
        action: 'Add skill levels in your profile.',
        category: 'skills' as const,
        priority: (needsSkills ? 'high' : 'medium') as 'high' | 'medium',
        impact_score: 4,
        metric: 'profile_completion',
      },
      {
        title: 'Keep QR active',
        description: 'Turn on your QR card before events so recruiters can save you instantly.',
        action: 'Toggle QR card to active.',
        category: 'readiness' as const,
        priority: (studentData.qr_active ? 'low' : 'medium') as 'low' | 'medium',
        impact_score: 3,
        metric: 'QR_scans',
      },
      {
        title: 'Add education highlight',
        description: 'Include one standout course, project, or GPA to strengthen academics.',
        action: 'Update education with a highlight.',
        category: 'profile' as const,
        priority: 'medium' as const,
        impact_score: 3,
        metric: 'recruiter_interest',
      },
      ...(missingSummary
        ? [{
            title: 'Finish basic info',
            description: 'Fill in your contact and headline details to appear complete.',
            action: 'Complete missing profile fields.',
            category: 'profile' as const,
            priority: 'high' as const,
            impact_score: 4,
            metric: 'profile_completion',
          }]
        : []),
    ];
  };

  const getPriorityBadgeClasses = (priority?: 'high' | 'medium' | 'low') => {
    switch (priority) {
      case 'high':
        return 'bg-red-100 text-red-700 border border-red-200';
      case 'low':
        return 'bg-gray-100 text-gray-700 border border-gray-200';
      default:
        return 'bg-amber-100 text-amber-700 border border-amber-200';
    }
  };

  const getTipIcon = (category?: string) => {
    switch (category) {
      case 'skills':
        return FiAward;
      case 'experience':
        return FiTrendingUp;
      case 'branding':
        return FiTarget;
      case 'readiness':
        return FiZap;
      default:
        return FiUser;
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
          <p className="mt-4 text-gray-600">Loading your dashboard...</p>
        </div>
      </div>
    );
  }

  if (error || !student) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="text-red-500 text-6xl mb-4">⚠️</div>
          <h1 className="text-2xl font-bold text-gray-900 mb-2">Dashboard Error</h1>
          <p className="text-gray-600 mb-4">{error || 'Failed to load dashboard'}</p>
          <button 
            onClick={fetchDashboardData}
            className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700"
          >
            Try Again
          </button>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-100">
      {/* Enhanced Header with Floating Elements */}
      <div className="relative overflow-hidden">
        <div className="absolute inset-0 bg-gradient-to-r from-blue-600/10 via-purple-600/10 to-indigo-600/10"></div>
        <div className="absolute top-10 left-10 w-32 h-32 bg-gradient-to-r from-blue-400/20 to-purple-400/20 rounded-full blur-3xl"></div>
        <div className="absolute top-20 right-20 w-24 h-24 bg-gradient-to-r from-indigo-400/20 to-pink-400/20 rounded-full blur-2xl"></div>
        
        <div className="relative max-w-7xl mx-auto px-6 py-12">
          {/* Ultra-Modern Welcome Header */}
          <div className="mb-12">
            <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between">
              <div className="flex items-center mb-6 lg:mb-0">
                <div className="relative">
                  <div className="w-16 h-16 bg-gradient-to-r from-blue-600 via-purple-600 to-indigo-600 rounded-2xl flex items-center justify-center shadow-xl">
                    <FiUser className="text-white text-2xl" />
                  </div>
                  <div className="absolute -inset-1 bg-gradient-to-r from-blue-600 via-purple-600 to-indigo-600 rounded-2xl blur-lg opacity-30 animate-pulse"></div>
                </div>
                <div className="ml-6">
                  <h1 className="text-4xl font-bold text-gray-900 mb-2">
                    Welcome back, <span className="bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent">{student.name || 'User'}</span>!
                  </h1>
                  <p className="text-xl text-gray-600">
                    Your professional journey continues here. Let's make today count! 🚀
                  </p>
                </div>
              </div>
              
              {/* Quick Action Buttons */}
              <div className="flex space-x-4">
                <Link
                  to="/profile"
                  className="group relative overflow-hidden bg-white/80 backdrop-blur-sm border border-white/50 text-gray-700 px-6 py-3 rounded-xl hover:bg-white hover:shadow-lg transition-all duration-300 flex items-center space-x-2"
                >
                  <FiEdit3 className="text-lg" />
                  <span className="font-medium">Edit Profile</span>
                </Link>
                <Link
                  to="/my-card"
                  className="group relative overflow-hidden bg-gradient-to-r from-blue-600 to-purple-600 text-white px-6 py-3 rounded-xl hover:from-blue-700 hover:to-purple-700 transition-all duration-300 shadow-lg hover:shadow-xl flex items-center space-x-2"
                >
                  <FiCode className="text-lg" />
                  <span className="font-medium">My QR Card</span>
                </Link>
              </div>
            </div>
          </div>

          {/* Enhanced Stats Cards Grid */}
          <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-4 gap-6 mb-12">
            {/* Profile Views Card */}
            <div className="group relative overflow-hidden bg-white/80 backdrop-blur-xl border border-white/40 rounded-2xl shadow-xl p-6 hover:shadow-2xl transition-all duration-300 hover:-translate-y-1">
              <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-blue-500 to-indigo-500"></div>
              <div className="flex items-center justify-between mb-4">
                <div className="w-12 h-12 bg-gradient-to-r from-blue-100 to-indigo-100 rounded-xl flex items-center justify-center group-hover:scale-110 transition-transform duration-300">
                  <FiEye className="text-blue-600 text-xl" />
                </div>
                <div className={`flex items-center space-x-1 text-sm font-medium ${formatPercentageChange(stats?.profile_views_change || 0).color}`}>
                  {formatPercentageChange(stats?.profile_views_change || 0).icon === 'up' && <FiArrowUp className="text-xs" />}
                  {formatPercentageChange(stats?.profile_views_change || 0).icon === 'down' && <FiArrowDown className="text-xs" />}
                  <span>{formatPercentageChange(stats?.profile_views_change || 0).text}</span>
                </div>
              </div>
              <div className="text-3xl font-bold text-gray-900 mb-1">
                {stats?.profile_views || 0}
              </div>
              <div className="text-gray-600 text-sm mb-2">Profile Views</div>
              <div className="text-xs text-gray-500">Last 30 days</div>
            </div>

            {/* QR Scans Card */}
            <div className="group relative overflow-hidden bg-white/80 backdrop-blur-xl border border-white/40 rounded-2xl shadow-xl p-6 hover:shadow-2xl transition-all duration-300 hover:-translate-y-1">
              <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-emerald-500 to-teal-500"></div>
              <div className="flex items-center justify-between mb-4">
                <div className="w-12 h-12 bg-gradient-to-r from-emerald-100 to-teal-100 rounded-xl flex items-center justify-center group-hover:scale-110 transition-transform duration-300">
                  <FiCode className="text-emerald-600 text-xl" />
                </div>
                <div className={`flex items-center space-x-1 text-sm font-medium ${formatPercentageChange(stats?.qr_scans_change || 0).color}`}>
                  {formatPercentageChange(stats?.qr_scans_change || 0).icon === 'up' && <FiArrowUp className="text-xs" />}
                  {formatPercentageChange(stats?.qr_scans_change || 0).icon === 'down' && <FiArrowDown className="text-xs" />}
                  <span>{formatPercentageChange(stats?.qr_scans_change || 0).text}</span>
                </div>
              </div>
              <div className="text-3xl font-bold text-gray-900 mb-1">
                {stats?.qr_scans || 0}
              </div>
              <div className="text-gray-600 text-sm mb-2">QR Scans</div>
              <div className="text-xs text-gray-500">Last 7 days</div>
            </div>

            {/* Shortlist Additions Card */}
            <div className="group relative overflow-hidden bg-white/80 backdrop-blur-xl border border-white/40 rounded-2xl shadow-xl p-6 hover:shadow-2xl transition-all duration-300 hover:-translate-y-1">
              <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-purple-500 to-pink-500"></div>
              <div className="flex items-center justify-between mb-4">
                <div className="w-12 h-12 bg-gradient-to-r from-purple-100 to-pink-100 rounded-xl flex items-center justify-center group-hover:scale-110 transition-transform duration-300">
                  <FiHeart className="text-purple-600 text-xl" />
                </div>
                <div className={`flex items-center space-x-1 text-sm font-medium ${formatPercentageChange(stats?.shortlist_additions_change || 0).color}`}>
                  {formatPercentageChange(stats?.shortlist_additions_change || 0).icon === 'up' && <FiArrowUp className="text-xs" />}
                  {formatPercentageChange(stats?.shortlist_additions_change || 0).icon === 'down' && <FiArrowDown className="text-xs" />}
                  <span>{formatPercentageChange(stats?.shortlist_additions_change || 0).text}</span>
                </div>
              </div>
              <div className="text-3xl font-bold text-gray-900 mb-1">
                {stats?.shortlist_additions || 0}
              </div>
              <div className="text-gray-600 text-sm mb-2">Shortlist Adds</div>
              <div className="text-xs text-gray-500">This month</div>
            </div>

            {/* Profile Completion Card */}
            <div className="group relative overflow-hidden bg-white/80 backdrop-blur-xl border border-white/40 rounded-2xl shadow-xl p-6 hover:shadow-2xl transition-all duration-300 hover:-translate-y-1">
              <div className={`absolute top-0 left-0 w-full h-1 ${
                student.profile_completion >= 80 ? 'bg-gradient-to-r from-green-500 to-emerald-500' :
                student.profile_completion >= 50 ? 'bg-gradient-to-r from-yellow-500 to-orange-500' : 
                'bg-gradient-to-r from-red-500 to-pink-500'
              }`}></div>
              <div className="flex items-center justify-between mb-4">
                <div className={`w-12 h-12 rounded-xl flex items-center justify-center group-hover:scale-110 transition-transform duration-300 ${
                  student.profile_completion >= 80 ? 'bg-gradient-to-r from-green-100 to-emerald-100' :
                  student.profile_completion >= 50 ? 'bg-gradient-to-r from-yellow-100 to-orange-100' : 
                  'bg-gradient-to-r from-red-100 to-pink-100'
                }`}>
                  <FiPieChart className={`text-xl ${
                    student.profile_completion >= 80 ? 'text-green-600' :
                    student.profile_completion >= 50 ? 'text-yellow-600' : 'text-red-600'
                  }`} />
                </div>
                <div className="flex items-center space-x-1 text-blue-600 text-sm font-medium">
                  <FiZap className="text-xs" />
                  <span>Boost</span>
                </div>
              </div>
              <div className={`text-3xl font-bold mb-1 ${getCompletionColor(student.profile_completion)}`}>
                {student.profile_completion}%
              </div>
              <div className="text-gray-600 text-sm mb-2">Complete</div>
              <div className="text-xs text-gray-500">
                {student.profile_completion >= 80 ? 'Excellent!' :
                 student.profile_completion >= 50 ? 'Good progress' : 'Keep going'}
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Enhanced Transition Section */}
      <div className="relative">
        {/* Gradient Separator */}
        <div className="absolute inset-0 bg-gradient-to-b from-gray-50 via-white/50 to-transparent h-24"></div>
        
        {/* Decorative Elements */}
        <div className="relative max-w-7xl mx-auto px-6 py-8">
          <div className="flex items-center justify-center space-x-4 mb-6">
            <div className="flex-1 h-px bg-gradient-to-r from-transparent via-gray-300 to-gray-300"></div>
            <div className="flex items-center space-x-2 px-4 py-2 bg-white/80 backdrop-blur-sm rounded-full border border-gray-200 shadow-sm">
              <div className="w-2 h-2 bg-blue-500 rounded-full animate-pulse"></div>
              <span className="text-sm font-medium text-gray-600">Your Dashboard</span>
              <div className="w-2 h-2 bg-purple-500 rounded-full animate-pulse delay-75"></div>
            </div>
            <div className="flex-1 h-px bg-gradient-to-l from-transparent via-gray-300 to-gray-300"></div>
          </div>
        </div>
      </div>

      {/* Main Content Area */}
      <div className="max-w-7xl mx-auto px-6 pb-12">
        <div className="grid xl:grid-cols-3 gap-8">
          {/* Main Content - Left Side */}
          <div className="xl:col-span-2 space-y-8">
            
            {/* Enhanced Profile Completion */}
            <div className={`relative overflow-hidden rounded-2xl border backdrop-blur-xl p-8 shadow-xl ${getCompletionBgColor(student.profile_completion)}`}>
              <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-blue-500 via-purple-500 to-indigo-500"></div>
              <div className="flex items-center justify-between mb-6">
                <div className="flex items-center">
                  <div className="w-12 h-12 bg-white/80 rounded-xl flex items-center justify-center mr-4">
                    <FiBarChart className="text-gray-700 text-xl" />
                  </div>
                  <div>
                    <h2 className="text-2xl font-bold text-gray-900">Profile Completion</h2>
                    <p className="text-gray-600">Complete your profile to maximize opportunities</p>
                  </div>
                </div>
                <div className="text-right">
                  <div className={`text-4xl font-bold ${getCompletionColor(student.profile_completion)}`}>
                    {student.profile_completion}%
                  </div>
                  <div className="text-sm text-gray-600">Complete</div>
                </div>
              </div>
              
              {/* Enhanced Progress Bar */}
              <div className="relative bg-white/50 rounded-full h-4 mb-6 overflow-hidden">
                <div
                  className={`h-full rounded-full transition-all duration-1000 ${
                    student.profile_completion >= 80 ? 'bg-gradient-to-r from-green-400 to-emerald-500' :
                    student.profile_completion >= 50 ? 'bg-gradient-to-r from-yellow-400 to-orange-500' : 
                    'bg-gradient-to-r from-red-400 to-pink-500'
                  }`}
                  style={{ width: `${student.profile_completion}%` }}
                >
                  <div className="absolute inset-0 bg-gradient-to-r from-white/20 to-transparent rounded-full"></div>
                </div>
              </div>
              
              {/* Profile Stats Grid */}
              <div className="grid grid-cols-3 gap-6 mb-6">
                <div className="text-center bg-white/60 rounded-xl p-4">
                  <div className="w-10 h-10 bg-gradient-to-r from-blue-500 to-indigo-500 rounded-lg flex items-center justify-center mx-auto mb-2">
                    <FiBriefcase className="text-white text-lg" />
                  </div>
                  <div className="text-2xl font-bold text-gray-900">{student.experiences_count}</div>
                  <div className="text-sm text-gray-600">Experiences</div>
                </div>
                <div className="text-center bg-white/60 rounded-xl p-4">
                  <div className="w-10 h-10 bg-gradient-to-r from-purple-500 to-pink-500 rounded-lg flex items-center justify-center mx-auto mb-2">
                    <FiBookOpen className="text-white text-lg" />
                  </div>
                  <div className="text-2xl font-bold text-gray-900">{student.educations_count}</div>
                  <div className="text-sm text-gray-600">Education</div>
                </div>
                <div className="text-center bg-white/60 rounded-xl p-4">
                  <div className="w-10 h-10 bg-gradient-to-r from-emerald-500 to-teal-500 rounded-lg flex items-center justify-center mx-auto mb-2">
                    <FiAward className="text-white text-lg" />
                  </div>
                  <div className="text-2xl font-bold text-gray-900">{student.skills_count}</div>
                  <div className="text-sm text-gray-600">Skills</div>
                </div>
              </div>
              
              {student.profile_completion < 100 && (
                <Link
                  to="/profile"
                  className="group relative overflow-hidden block w-full text-center bg-white text-gray-800 py-4 px-6 rounded-xl hover:bg-gray-50 transition-all duration-300 font-semibold shadow-lg hover:shadow-xl"
                >
                  <div className="flex items-center justify-center space-x-2">
                    <FiEdit3 className="text-lg" />
                    <span>Complete Your Profile</span>
                    <FiArrowUp className="text-sm group-hover:translate-x-1 transition-transform duration-300" />
                  </div>
                </Link>
              )}
            </div>

            {/* Enhanced Quick Actions Grid */}
            <div className="bg-white/80 backdrop-blur-xl border border-white/40 rounded-2xl shadow-xl p-8">
              <div className="flex items-center justify-between mb-8">
                <div className="flex items-center">
                  <div className="w-12 h-12 bg-gradient-to-r from-orange-500 to-red-500 rounded-xl flex items-center justify-center mr-4">
                    <FiZap className="text-white text-xl" />
                  </div>
                  <div>
                    <h2 className="text-2xl font-bold text-gray-900">Quick Actions</h2>
                    <p className="text-gray-600">Manage your professional presence</p>
                  </div>
                </div>
                <div className="flex items-center space-x-2 text-sm text-gray-500">
                  <FiClock className="text-xs" />
                  <span>Updated {new Date().toLocaleDateString()}</span>
                </div>
              </div>
              
              <div className="grid md:grid-cols-2 gap-6">
                {/* Edit Profile Action */}
                <Link
                  to="/profile"
                  className="group relative overflow-hidden bg-gradient-to-br from-blue-50 to-indigo-50 hover:from-blue-100 hover:to-indigo-100 border border-blue-200 rounded-2xl p-6 transition-all duration-300 hover:shadow-lg hover:-translate-y-1"
                >
                  <div className="flex items-center mb-4">
                    <div className="w-12 h-12 bg-gradient-to-r from-blue-500 to-indigo-500 rounded-xl flex items-center justify-center mr-4 group-hover:scale-110 transition-transform duration-300">
                      <FiEdit3 className="text-white text-xl" />
                    </div>
                    <div>
                      <h3 className="font-bold text-gray-900 text-lg">Edit Profile</h3>
                      <p className="text-sm text-gray-600">Update your information</p>
                    </div>
                  </div>
                  <div className="text-xs text-blue-600 font-medium">Click to manage →</div>
                  <div className="absolute inset-0 bg-gradient-to-r from-blue-500/5 to-indigo-500/5 opacity-0 group-hover:opacity-100 transition-opacity duration-300 rounded-2xl"></div>
                </Link>
                
                {/* QR Card Action */}
                <Link
                  to="/my-card"
                  className="group relative overflow-hidden bg-gradient-to-br from-emerald-50 to-teal-50 hover:from-emerald-100 hover:to-teal-100 border border-emerald-200 rounded-2xl p-6 transition-all duration-300 hover:shadow-lg hover:-translate-y-1"
                >
                  <div className="flex items-center mb-4">
                    <div className="w-12 h-12 bg-gradient-to-r from-emerald-500 to-teal-500 rounded-xl flex items-center justify-center mr-4 group-hover:scale-110 transition-transform duration-300">
                      <FiCode className="text-white text-xl" />
                    </div>
                    <div>
                      <h3 className="font-bold text-gray-900 text-lg">My QR Card</h3>
                      <p className="text-sm text-gray-600">View & share your card</p>
                    </div>
                  </div>
                  <div className="text-xs text-emerald-600 font-medium">Access now →</div>
                  <div className="absolute inset-0 bg-gradient-to-r from-emerald-500/5 to-teal-500/5 opacity-0 group-hover:opacity-100 transition-opacity duration-300 rounded-2xl"></div>
                </Link>
                
                {/* Public Profile Action */}
                <a
                  href={`/p/${student.id}`}
                  target="_blank"
                  rel="noopener noreferrer"
                  className="group relative overflow-hidden bg-gradient-to-br from-purple-50 to-pink-50 hover:from-purple-100 hover:to-pink-100 border border-purple-200 rounded-2xl p-6 transition-all duration-300 hover:shadow-lg hover:-translate-y-1"
                >
                  <div className="flex items-center mb-4">
                    <div className="w-12 h-12 bg-gradient-to-r from-purple-500 to-pink-500 rounded-xl flex items-center justify-center mr-4 group-hover:scale-110 transition-transform duration-300">
                      <FiExternalLink className="text-white text-xl" />
                    </div>
                    <div>
                      <h3 className="font-bold text-gray-900 text-lg">Public Profile</h3>
                      <p className="text-sm text-gray-600">See how others view you</p>
                    </div>
                  </div>
                  <div className="text-xs text-purple-600 font-medium">Open preview →</div>
                  <div className="absolute inset-0 bg-gradient-to-r from-purple-500/5 to-pink-500/5 opacity-0 group-hover:opacity-100 transition-opacity duration-300 rounded-2xl"></div>
                </a>
              </div>
            </div>

            {/* Enhanced Recent Activity */}
            {stats?.recent_activity && stats.recent_activity.length > 0 && (
              <div className="bg-white/80 backdrop-blur-xl border border-white/40 rounded-2xl shadow-xl p-8">
                <div className="flex items-center justify-between mb-8">
                  <div className="flex items-center">
                    <div className="w-12 h-12 bg-gradient-to-r from-indigo-500 to-blue-500 rounded-xl flex items-center justify-center mr-4">
                      <FiActivity className="text-white text-xl" />
                    </div>
                    <div>
                      <h2 className="text-2xl font-bold text-gray-900">Recent Activity</h2>
                      <p className="text-gray-600">Track your professional engagement</p>
                    </div>
                  </div>
                  <div className="flex items-center space-x-2">
                    <button className="p-2 text-gray-400 hover:text-gray-600 transition-colors">
                      <FiFilter className="text-lg" />
                    </button>
                    <button className="p-2 text-gray-400 hover:text-gray-600 transition-colors">
                      <FiMoreHorizontal className="text-lg" />
                    </button>
                  </div>
                </div>
                <div className="space-y-4">
                  {stats.recent_activity.map((activity, index) => (
                    <div key={activity.id} className="group relative overflow-hidden bg-gradient-to-r from-gray-50 to-white hover:from-blue-50 hover:to-indigo-50 rounded-xl p-4 border border-gray-100 hover:border-blue-200 transition-all duration-300">
                      <div className="flex items-center justify-between">
                        <div className="flex items-center space-x-4">
                          <div className={`w-10 h-10 rounded-xl flex items-center justify-center ${
                            index % 3 === 0 ? 'bg-gradient-to-r from-blue-100 to-indigo-100' :
                            index % 3 === 1 ? 'bg-gradient-to-r from-emerald-100 to-teal-100' :
                            'bg-gradient-to-r from-purple-100 to-pink-100'
                          }`}>
                            <FiTrendingUp className={`text-lg ${
                              index % 3 === 0 ? 'text-blue-600' :
                              index % 3 === 1 ? 'text-emerald-600' :
                              'text-purple-600'
                            }`} />
                          </div>
                          <div>
                            <div className="font-semibold text-gray-900">
                              {activity.action || 'Activity'}
                            </div>
                            <div className="text-sm text-gray-600">Professional activity</div>
                          </div>
                        </div>
                        <div className="flex items-center space-x-3 text-sm text-gray-500">
                          <div className="flex items-center space-x-1">
                            <FiCalendar className="text-xs" />
                            <span>{new Date(activity.created_at).toLocaleDateString()}</span>
                          </div>
                          <div className="flex items-center space-x-1">
                            <FiClock className="text-xs" />
                            <span>{new Date(activity.created_at).toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'})}</span>
                          </div>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            )}
          </div>

          {/* Enhanced Sidebar */}
          <div className="space-y-8">
            {/* Ultra-Modern QR Card Status */}
            <div className="relative overflow-hidden bg-white/80 backdrop-blur-xl border border-white/40 rounded-2xl shadow-xl p-6">
              <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-emerald-500 to-teal-500"></div>
              <div className="flex items-center mb-6">
                <div className="w-10 h-10 bg-gradient-to-r from-emerald-500 to-teal-500 rounded-xl flex items-center justify-center mr-3">
                  <FiCode className="text-white text-lg" />
                </div>
                <h3 className="text-xl font-bold text-gray-900">QR Card Status</h3>
              </div>
              
              <div className={`relative overflow-hidden rounded-xl p-6 ${student.qr_active ? 'bg-gradient-to-br from-green-50 to-emerald-50 border border-green-200' : 'bg-gradient-to-br from-yellow-50 to-orange-50 border border-yellow-200'}`}>
                <div className="flex items-center mb-4">
                  <div className={`w-12 h-12 rounded-xl flex items-center justify-center mr-4 ${
                    student.qr_active ? 'bg-gradient-to-r from-green-500 to-emerald-500' : 'bg-gradient-to-r from-yellow-500 to-orange-500'
                  }`}>
                    {student.qr_active ? (
                      <FiCheckCircle className="text-white text-xl" />
                    ) : (
                      <FiXCircle className="text-white text-xl" />
                    )}
                  </div>
                  <div>
                    <div className="text-xl font-bold text-gray-900">
                      {student.qr_active ? 'Active & Ready' : 'Inactive'}
                    </div>
                    <div className="text-sm text-gray-600">
                      {student.qr_active ? 'Recruiters can scan your card' : 'Limited functionality'}
                    </div>
                  </div>
                </div>
                
                {/* Enhanced Card Status Mini Card */}
                <div className="bg-white/60 backdrop-blur-sm rounded-xl p-4 mb-4 border border-white/50">
                  <div className="grid grid-cols-2 gap-4">
                    {/* Card Status Side */}
                    <div className="text-center">
                      <div className={`w-8 h-8 mx-auto mb-2 rounded-lg flex items-center justify-center ${
                        student.qr_active ? 'bg-green-500' : 'bg-yellow-500'
                      }`}>
                        {student.qr_active ? (
                          <FiCheckCircle className="text-white text-sm" />
                        ) : (
                          <FiClock className="text-white text-sm" />
                        )}
                      </div>
                      <div className="text-xs text-gray-600 mb-1">Status</div>
                      <div className={`text-sm font-bold ${student.qr_active ? 'text-green-600' : 'text-yellow-600'}`}>
                        {student.qr_active ? 'LIVE' : 'PENDING'}
                      </div>
                    </div>
                    
                    {/* Recent Scans Side */}
                    <div className="text-center">
                      <div className="w-8 h-8 mx-auto mb-2 bg-gradient-to-r from-blue-500 to-purple-500 rounded-lg flex items-center justify-center">
                        <FiEye className="text-white text-sm" />
                      </div>
                      <div className="text-xs text-gray-600 mb-1">Scans (7d)</div>
                      <div className="text-sm font-bold text-gray-900">
                        {stats?.qr_scans || 0}
                      </div>
                    </div>
                  </div>
                  
                  {/* Mini Activity Indicator */}
                  <div className="mt-4 pt-3 border-t border-gray-200/50">
                    <div className="flex items-center justify-between text-xs">
                      <span className="text-gray-500">Last scan:</span>
                      <span className="text-gray-700 font-medium">
                        {stats?.qr_scans && stats.qr_scans > 0 ? 'Recent' : 'No activity'}
                      </span>
                    </div>
                    {stats?.qr_scans && stats.qr_scans > 0 && (
                      <div className="mt-2">
                        <div className="w-full bg-gray-200 rounded-full h-1.5">
                          <div 
                            className="bg-gradient-to-r from-blue-500 to-purple-500 h-1.5 rounded-full transition-all duration-500" 
                            style={{ width: `${Math.min((stats.qr_scans / 10) * 100, 100)}%` }}
                          ></div>
                        </div>
                        <div className="text-xs text-gray-500 mt-1">Activity level</div>
                      </div>
                    )}
                  </div>
                </div>
                
                <Link
                  to="/my-card"
                  className="block w-full text-center bg-white text-gray-800 py-3 px-4 rounded-xl hover:bg-gray-50 transition-all duration-300 font-semibold shadow-lg hover:shadow-xl"
                >
                  Manage QR Card
                </Link>
              </div>
            </div>

            {/* Enhanced Profile Summary */}
            <div className="relative overflow-hidden bg-white/80 backdrop-blur-xl border border-white/40 rounded-2xl shadow-xl p-6">
              <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-purple-500 to-pink-500"></div>
              <div className="flex items-center mb-6">
                <div className="w-10 h-10 bg-gradient-to-r from-purple-500 to-pink-500 rounded-xl flex items-center justify-center mr-3">
                  <FiUser className="text-white text-lg" />
                </div>
                <h3 className="text-xl font-bold text-gray-900">Profile Summary</h3>
              </div>
              
              <div className="text-center mb-6">
                <div className="relative inline-block">
                  {student.photo_url ? (
                    <img
                      src={student.photo_url}
                      alt={student.name || 'User'}
                      className="w-20 h-20 rounded-2xl object-cover border-4 border-white shadow-lg"
                    />
                  ) : (
                    <div className="w-20 h-20 rounded-2xl bg-gradient-to-r from-blue-500 to-purple-600 flex items-center justify-center text-white text-2xl font-bold border-4 border-white shadow-lg">
                      {student.name?.charAt(0)?.toUpperCase() || 'U'}
                    </div>
                  )}
                  <div className="absolute -bottom-1 -right-1 w-6 h-6 bg-green-500 rounded-full border-2 border-white flex items-center justify-center">
                    <FiCheckCircle className="text-white text-xs" />
                  </div>
                </div>
                
                <h4 className="font-bold text-gray-900 text-lg mt-4">{student.name || 'Unknown User'}</h4>
                <p className="text-gray-600 text-sm">{student.email}</p>
                
                {student.degree && student.university && (
                  <div className="mt-3 p-3 bg-gradient-to-r from-blue-50 to-indigo-50 rounded-xl">
                    <p className="text-sm font-semibold text-gray-900">
                      {student.degree}
                      {student.graduation_year && ` (${student.graduation_year})`}
                    </p>
                    <p className="text-xs text-gray-600 flex items-center justify-center mt-1">
                      <FiMapPin className="mr-1" />
                      {student.university}
                    </p>
                  </div>
                )}
              </div>
              
              <div className="grid grid-cols-2 gap-3 mb-6">
                <Link
                  to="/profile"
                  className="group bg-gradient-to-r from-blue-600 to-purple-600 text-white py-3 px-4 rounded-xl hover:from-blue-700 hover:to-purple-700 transition-all duration-300 text-center font-semibold shadow-lg hover:shadow-xl flex items-center justify-center space-x-2"
                >
                  <FiEdit3 className="text-sm" />
                  <span>Edit</span>
                </Link>
                <button className="group bg-white text-gray-700 py-3 px-4 rounded-xl hover:bg-gray-50 transition-all duration-300 border border-gray-200 font-semibold shadow-lg hover:shadow-xl flex items-center justify-center space-x-2">
                  <FiShare2 className="text-sm" />
                  <span>Share</span>
                </button>
              </div>
            </div>

            {/* Enhanced Pro Tips */}
            <div className="relative overflow-hidden bg-gradient-to-br from-blue-50 to-indigo-100 rounded-2xl p-6 border border-blue-200 shadow-xl">
              <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-blue-500 to-indigo-500"></div>
              <div className="flex items-center justify-between mb-6">
                <div className="flex items-center">
                  <div className="w-10 h-10 bg-gradient-to-r from-blue-500 to-indigo-500 rounded-xl flex items-center justify-center mr-3">
                    <FiTarget className="text-white text-lg" />
                  </div>
                  <div>
                    <h3 className="text-xl font-bold text-blue-900">Pro Tips</h3>
                    <p className="text-sm text-blue-800">AI-powered, actionable nudges for your profile</p>
                  </div>
                </div>
                <div className="text-xs px-3 py-1 rounded-full bg-white/70 border border-blue-200 text-blue-800 font-semibold">
                  {tipsStatus === 'loading' ? 'Generating…' : tipsStatus === 'fallback' ? 'Fallback' : tipsStatus === 'error' ? 'Offline mode' : 'AI-powered'}
                </div>
              </div>
              
              <div className="space-y-4">
                {tipsStatus === 'loading' && [1, 2, 3].map((item) => (
                  <div key={item} className="bg-white/60 rounded-xl p-4 animate-pulse">
                    <div className="h-4 bg-blue-100 rounded w-2/3 mb-2"></div>
                    <div className="h-3 bg-blue-100 rounded w-5/6"></div>
                  </div>
                ))}

                {tipsStatus !== 'loading' && proTips.length === 0 && (
                  <div className="bg-white/70 rounded-xl p-4 border border-dashed border-blue-200 text-sm text-blue-900">
                    <div className="flex items-center justify-between">
                      <span>No tips yet. Try refreshing.</span>
                      <button
                        onClick={fetchDashboardData}
                        className="text-blue-700 hover:text-blue-900 font-semibold"
                      >
                        Refresh
                      </button>
                    </div>
                  </div>
                )}

                {proTips.map((tip, index) => {
                  const Icon = getTipIcon(tip.category);
                  const impactWidth = Math.min(((tip.impact_score ?? 3) / 5) * 100, 100);

                  return (
                    <div key={index} className="group bg-white/80 rounded-xl p-4 hover:bg-white transition-all duration-300 border border-white/60">
                      <div className="flex items-start justify-between">
                        <div className="flex items-start space-x-3">
                          <div className="w-10 h-10 bg-blue-100 rounded-xl flex items-center justify-center mt-0.5">
                            <Icon className="text-blue-600 text-lg" />
                          </div>
                          <div>
                            <p className="text-sm font-semibold text-blue-900">{tip.title}</p>
                            <p className="text-sm text-blue-800 mt-1">{tip.description}</p>
                            {tip.action && (
                              <p className="text-xs text-blue-700 font-medium mt-2">Next: {tip.action}</p>
                            )}
                          </div>
                        </div>
                        <span className={`text-[11px] px-2 py-1 rounded-full font-semibold ${getPriorityBadgeClasses(tip.priority)}`}>
                          {(tip.priority || 'medium').toUpperCase()} PRIORITY
                        </span>
                      </div>

                      <div className="mt-3">
                        <div className="flex items-center justify-between text-[11px] text-blue-800 mb-1">
                          <span className="uppercase tracking-wide">{tip.category || 'profile'}</span>
                          <span>Impact {tip.impact_score ?? 3}/5 • {tip.metric || 'profile_completion'}</span>
                        </div>
                        <div className="bg-blue-200 rounded-full h-2 overflow-hidden">
                          <div
                            className="bg-gradient-to-r from-blue-500 to-indigo-500 h-2"
                            style={{ width: `${impactWidth}%` }}
                          ></div>
                        </div>
                      </div>
                    </div>
                  );
                })}
              </div>
              
              {tipsMessage && (
                <div className="mt-6 p-4 bg-white/80 rounded-xl border border-blue-200 text-sm text-blue-800">
                  {tipsMessage}
                </div>
              )}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default StudentDashboard;
